package com.aote.workflow.perform;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.JarURLConnection;
import java.net.URL;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import org.apache.log4j.Logger;

import com.aote.ThreadResource;

/**
 * 流程图管理器
 */
public class DiagramManager {

    static Logger log = Logger.getLogger(DiagramManager.class);

    /**
     *
     */
    private static final long serialVersionUID = 373843474314241569L;

    private static DiagramManager instance = new DiagramManager();

    public static DiagramManager getInstance() {
        return instance;
    }

    /**
     * 流程图名字与内容列表
     */
    private Hashtable nameAndDiagrams = new Hashtable();

    /**
     * 构造管理器
     */
    public DiagramManager() {
        load();
    }

    /**
     * 装载所有流程图, 在processes下
     */
    public void load() {
        URL url = this.getClass().getClassLoader().getResource("processes/");
        // 流程定义目录不存在，跳过
        if (url == null) {
            return;
        }
        String path = url.getPath();
        File[] files = new File(path).listFiles();
        for (File f : files) {
            // 得到文件名
            String fileName = f.getName();
            if (fileName == null) {
                continue;
            }
            try {
                // 生成流程图
                String digName = fileName.substring(0, fileName.indexOf("."));
                Diagram diagram = new Diagram();
                diagram.setName(digName);
                nameAndDiagrams.put(digName, diagram);
                // 根据文件路径得到流
                String result = "";
                String encoding = "UTF-8";
                InputStreamReader read = new InputStreamReader(
                        new FileInputStream(f), encoding);// 考虑到编码格式
                BufferedReader bufferedReader = new BufferedReader(read);
                String lineTxt = null;
                while ((lineTxt = bufferedReader.readLine()) != null) {
                    result += lineTxt;
                }
                bufferedReader.close();
                // 设置流程图内容
                diagram.setContext(result);
            } catch (Exception e) {
                log.error("流程" + fileName + ": " + e.getMessage());
            }
        }
    }

    /**
         装载登录人目录下流程图, 在/(登录人目录)/processes下
     */
    public  Diagram userLoad(String userUrl) {
        URL url = this.getClass().getClassLoader().getResource(userUrl);
        String path = url.getPath();
        File file = new File(path);
        String fileName = file.getName();
        try {
            // 生成流程图
            String digName = fileName.substring(0, fileName.indexOf("."));
            Diagram diagram = new Diagram();
            diagram.setName(digName);
            // 根据文件路径得到流
            String result = "";
            String encoding = "UTF-8";
            InputStreamReader read = new InputStreamReader(
                    new FileInputStream(file), encoding);// 考虑到编码格式
            BufferedReader bufferedReader = new BufferedReader(read);
            String lineTxt = null;
            while ((lineTxt = bufferedReader.readLine()) != null) {
                result += lineTxt;
            }
            bufferedReader.close();
            // 设置流程图内容
            diagram.setContext(result);
            return diagram;
        } catch (Exception e) {
            throw new RuntimeException("流程" + fileName + ": " + e.getMessage(), e);
        }
    }

    /**
     * 该方法提供将 jar包的对应资源文件夹中所有的资源文件获取并读成字符串集合
     *
     * @param url  url路径，通过this.getClass().getClassLoader().getResource(path)获取jar包文件路径
     * @param path 需要遍历的文件夹名称--后面记得加上/右斜杠
     * @return 读取到的文件内容---一个字符串是一个文件的内容
     */
    private List<String> loadResources(URL url, String path) {
        List<String> strs = new ArrayList<>();
        int statu = 0;          //状态分0：未读 1：正在读 2：已读
        try {
            System.out.println(url);
//            String jarPath = url.toString().substring(0, url.toString().indexOf("!/") + 2);
//            System.out.println(jarPath);
            URL jarURL = new URL(url.toString().substring(0, url.toString().indexOf("!/") + 2));
            JarURLConnection jarCon = (JarURLConnection) jarURL.openConnection();
            JarFile jarFile = jarCon.getJarFile();
            Enumeration<JarEntry> jarEntrys = jarFile.entries();
            while (jarEntrys.hasMoreElements()) {
                if (statu == 2)    //当状态为已读时，则关掉循环，避免报错
                    break;
                JarEntry entry = null;
                try {
                    entry = jarEntrys.nextElement();
                } catch (Exception e) {
                    break;
                }
                String name = entry.getName();
                if (name.startsWith(path) && !entry.isDirectory()) {
                    if (statu == 0)
                        statu = 1;      //将状态更改为正在读
                    BufferedInputStream biput = null;
                    try {
                        byte[] buffer = new byte[(int) entry.getSize()];
                        biput = new BufferedInputStream(this.getClass().getClassLoader().getResourceAsStream(name));
                        biput.read(buffer, 0, buffer.length);
                        String str = new String(buffer, "UTF-8");
                        strs.add(str);
                    } catch (Exception e) {
                        e.printStackTrace();
                    } finally {
                        if (biput != null)
                            biput.close();
                    }
                } else if (statu == 1) {
                    statu = 2;
                }
            }

        } catch (IOException e) {
            e.printStackTrace();
        }
        return strs;
    }

    /**
     * 得到所有流程图
     *
     * @return 流程图列表
     * @throws RemoteException
     */
    public Enumeration getDiagrams() throws RemoteException {
        return nameAndDiagrams.elements();
    }

    /**
     * 取的所有流程图的名字
     *
     * @return 所有流程图的名字
     */
    public LinkedList getDiagramNames() throws RemoteException {
        return new LinkedList(nameAndDiagrams.keySet());
    }

    /**
     * 根据流程图名字取的流程图
     *
     * @param name 流程图名字
     * @return 流程图
     */
    public Diagram getDiagramByName(String name) throws RemoteException {
        return (Diagram) nameAndDiagrams.get(name);
    }

    /**
     * 取登录用户的特殊流程图
     *
     * @param name  流程文件名
     * @return diagram 返回流程图
     */
    public Diagram getUserDiagramByName(String name) {
        // 从线程变量里获取登录用户目录名
        String dir = ThreadResource.ComponentDir.get();
        if (dir == null) {
            return null;
        }

        // 看对应目录下有没有给定内容
        String path = dir + "/processes/" + name + ".xml";
        if (ProcessDefManager.class.getClassLoader().getResourceAsStream(path) == null) {
            return null;
        }

        Diagram diagram = this.userLoad(path);
        return diagram;
    }
}