package com.aote.util;

import org.bouncycastle.util.encoders.Base64;
import org.springframework.util.Assert;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.util.zip.*;


/**
 * 对账文件处理工具类
 *
 * @author Qu Dihuai
 *
 */
public final class FileUtils
{
	/**
	 * Represents the end-of-file (or stream).
	 */
	public static final int EOF = -1;
	/**
	 * The default buffer size ({@value}) to use for
	 */
	private static final int DEFAULT_BUFFER_SIZE = 1024 * 4;

	/**
	 * 将txt格式的对账文件压缩为zip文件
	 *
	 * @param sourceFile
	 *                      源文件txt文件路径
	 * @param targetFile
	 *                      压缩生成的zip文件路径
	 * @return 压缩生成的zip文件路径
	 * @throws IOException
	 */
	public static String zip(final String sourceFile, final String targetFile) throws IOException
	{
		Assert.hasText(sourceFile, "源文件不能为空");
		Assert.hasText(targetFile, "目标文件不能为空");

		final File source = new File(sourceFile);

		Assert.isTrue(source.isFile(), "源不是文件");

		int len = -1;
		final byte[] buf = new byte[DEFAULT_BUFFER_SIZE];

		try (final ZipOutputStream zos = new ZipOutputStream(new FileOutputStream(targetFile));
				final BufferedOutputStream bos = new BufferedOutputStream(zos);
				final FileInputStream fis = new FileInputStream(source);
				final BufferedInputStream bis = new BufferedInputStream(fis))
		{
			final ZipEntry entry = new ZipEntry(source.getName());
			zos.putNextEntry(entry);

			while ((len = bis.read(buf)) != EOF)
			{
				bos.write(buf, 0, len);
			}

			bos.flush();
		}

		return targetFile;
	}

	/**
	 * 将zip格式的对账文件解压为txt文件
	 *
	 * @param sourceFile
	 *                           zip文件路径
	 * @param targetDirectory
	 *                           解压生成的txt文件所在文件夹目录
	 * @return 解压生成的txt文件路径
	 * @throws IOException
	 */
	public static String unzip(final String sourceFile, final String targetDirectory) throws IOException
	{
		Assert.hasText(sourceFile, "源文件不能为空");
		Assert.hasText(targetDirectory, "目标文件夹不能为空");

		final File source = new File(sourceFile);
		Assert.isTrue(source.isFile(), "源不是文件");

		final File target = new File(targetDirectory);
		if (target.exists())
		{
			target.delete();
			target.getParentFile().mkdirs();
			target.createNewFile();
		}

		try (final InputStream is = new FileInputStream(source);
				final ZipInputStream zis = new ZipInputStream(is);
				final BufferedInputStream bin = new BufferedInputStream(zis))
		{
			ZipEntry entry = null;

			while ((entry = zis.getNextEntry()) != null && !entry.isDirectory())
			{
				final File file = new File(targetDirectory, entry.getName());
				if (!file.exists())
				{
					file.getParentFile().mkdirs();
				}

				int len;
				byte[] buf = new byte[DEFAULT_BUFFER_SIZE];

				try (final BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(file)))
				{
					while ((len = zis.read(buf)) != EOF)
					{
						bos.write(buf, 0, len);
					}
					bos.flush();
				}
			}
		}

		return targetDirectory;
	}

	/**
	 * 功能：解析接口收到的fileContent字符串并落地(解base64，解压缩并落地)<br>
	 *
	 * @param filePath
	 *                       文件路径
	 * @param fileContent
	 *                       文件内容
	 * @return zip文件
	 * @throws Exception
	 */
	public static String decodeFileContent(final String filePath, final String fileContent) throws Exception
	{
		Assert.hasText(filePath, "文件路径不能为空");
		Assert.hasText(fileContent, "文件内容不能为空");

		FileOutputStream out = null;
		try
		{
			final byte[] encodedFileContentBytes = fileContent.getBytes(StandardCharsets.UTF_8);
			final byte[] fileContentBytes = Base64.decode(encodedFileContentBytes);
			final byte[] fileArray = inflater(fileContentBytes);

			final File file = new File(filePath);
			if (file.exists())
			{
				file.delete();
			}

			final File parentDirectory = file.getParentFile();
			if (!parentDirectory.exists())
			{
				parentDirectory.mkdirs();
			}

			file.createNewFile();
			out = new FileOutputStream(file);
			out.write(fileArray, 0, fileArray.length);
			out.flush();

			return filePath;
		}
		catch (final IOException | DataFormatException e)
		{
			throw e;
		}
		finally
		{
			closeQuietly(out);
		}
	}

	/**
	 * 对文件流进行压缩、编码
	 *
	 * @param filePath
	 *                    源txt文件的路径
	 * @return 压缩，Base64编码后的字符串
	 * @throws IOException
	 */
	public static String encodeFileContent(final String filePath) throws IOException
	{
		Assert.hasText(filePath, "文件路径不能为空");

		try (final FileInputStream fis = new FileInputStream(filePath);
				final ByteArrayOutputStream baos = new ByteArrayOutputStream())
		{
			final byte[] buffer = new byte[DEFAULT_BUFFER_SIZE];
			int n = 0;
			while (EOF != (n = fis.read(buffer)))
			{
				baos.write(buffer, 0, n);
			}

			final byte[] bytes = baos.toByteArray();
			final byte[] fileArray = deflater(bytes);
			final byte[] fileContentBytes = Base64.encode(fileArray);
			return new String(fileContentBytes, StandardCharsets.UTF_8);
		}
	}

	/**
	 * 解压缩.
	 *
	 * @param inputBytes
	 *                      byte[]数组类型的数据
	 * @return 解压缩后的数据
	 * @throws DataFormatException
	 */
	public static byte[] inflater(final byte[] inputBytes) throws DataFormatException
	{
		final Inflater compresser = new Inflater(false);
		compresser.setInput(inputBytes, 0, inputBytes.length);

		final byte[] result = new byte[1024];
		int compressedDataLength = 0;

		ByteArrayOutputStream baos = new ByteArrayOutputStream(inputBytes.length);
		try
		{
			baos = new ByteArrayOutputStream(inputBytes.length);
			while (!compresser.finished())
			{
				compressedDataLength = compresser.inflate(result);
				if (compressedDataLength == 0)
				{
					break;
				}
				baos.write(result, 0, compressedDataLength);
			}

			final byte[] bytes = baos.toByteArray();
			return bytes;
		}
		finally
		{
			compresser.end();
			closeQuietly(baos);
		}
	}

	/**
	 * 压缩.
	 *
	 * @param inputBytes
	 *                      需要解压缩的byte[]数组
	 * @return 压缩后的数据
	 */
	public static byte[] deflater(final byte[] inputBytes)
	{
		final Deflater compresser = new Deflater();
		compresser.setInput(inputBytes);
		compresser.finish();

		final byte[] result = new byte[1024];
		int compressedDataLength = 0;

		ByteArrayOutputStream baos = null;
		try
		{
			baos = new ByteArrayOutputStream(inputBytes.length);
			while (!compresser.finished())
			{
				compressedDataLength = compresser.deflate(result);
				baos.write(result, 0, compressedDataLength);
			}

			final byte[] bytes = baos.toByteArray();
			return bytes;
		}
		finally
		{
			compresser.end();
			closeQuietly(baos);
		}
	}

	private static void closeQuietly(final Closeable closeable)
	{
		try
		{
			if (closeable != null)
			{
				closeable.close();
			}
		}
		catch (final IOException ioe)
		{
			// ignore
		}
	}
}
