package com.aote.pay.xianbank.tongchuan;

import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.DateUtil;
import com.aote.logic.LogicServer;
import com.aote.pay.PaySuper;
import com.aote.sql.SqlServer;
import com.aote.util.PayUtil;
import com.aote.weixin.Config;
import lombok.extern.slf4j.Slf4j;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.nio.charset.StandardCharsets;
import java.util.Map;

/**
 * @Author 李魁艳
 * @Date 2021/7/13 18:46
 * @Version 1.0
 * @describe 西银惠付扫码
 * @filiale 铜川
 */
@Slf4j
@Component
public class MicroPayTongChuan implements PaySuper {

    @Autowired
    private LogicServer logicServer;

    @Autowired
    private SqlServer sqlServer;
    

    /**
     * 下单接口
     * @param json
     * @return 下单结果参数
     */
    @Override
    public String prePay(JSONObject json) {
        JSONObject result = new JSONObject();
        try {
            String money = String.valueOf(json.get("money"));
            String auth_code = String.valueOf(json.get("auth_code"));
            String attach = String.valueOf(json.get("attach"));
            String filiale = String.valueOf(json.get("filiale"));
            String doorId = String.valueOf(json.get("doorId"));
            String openplAppId = String.valueOf(json.get("openplAppId"));

            if (filiale == null || filiale.length() == 0) {
                throw new RuntimeException("公司信息不能为空！");
            }
            if (Double.parseDouble(money) <= 0) {
                throw new RuntimeException("支付金额必须大于0");
            }
            if ("".equals(auth_code)) {
                throw new RuntimeException("付款码不能为空!");
            }
            if ("".equals(openplAppId)) {
                throw new RuntimeException("开放平台唯一标识不能为空!");
            }
            if ("".equals(doorId)) {
                throw new RuntimeException("门店编号不能为空!");
            }
            JSONObject wxConfig = Config.getConfig(filiale);

            // 订单号
            String outTradeNo = PayUtil.getOrderNum(openplAppId);
            // 订单金额，单位为分
            String totalFee = String.valueOf(PayUtil.yuan2FenInt(money));
            // 签名
            JSONObject jsonString = new JSONObject();
            jsonString.put("OPENPL_APPID", openplAppId);
            jsonString.put("OUT_TRADE_NO", outTradeNo);
            jsonString.put("DOOR_ID", doorId);
            jsonString.put("TOTAL_FEE", totalFee);
            jsonString.put("AUTH_CODE", auth_code);
            jsonString.put("ATTATCH", attach);
            log.debug("签名参数: {}", jsonString);

            // 私钥
            String privateKey = wxConfig.getString("privateKey");
            // 用户密钥
            String openplAppsecret = wxConfig.getString("openpl_appsecret");

            // 加密并格式化发送消息
            String postMessage = FormatUtils.encryptMessageByPrivateKey(openplAppsecret, privateKey, String.valueOf(jsonString));
            // 下单地址
            String microPayUrl = wxConfig.getString("MicroPayUrl");
            log.debug("西银惠付付款码下单地址: {},下单参数: {}", microPayUrl, jsonString);
            byte[] data = null;
            try {
                data = HttpNetProvider.doPostBytes(microPayUrl, postMessage, "application/json");
            } catch (Exception e) {
                log.debug("请求响应错误");
                result.put("result_msg", "请求响应错误");
                result.put("err_msg", e.getMessage());
                throw new RuntimeException("付款码下单请求西银惠付未正常响应");
            }
            if (data != null) {
                String returnMessage = new String(data, StandardCharsets.UTF_8);
                Map<String, String> response = FormatUtils.handleReturnMessage(openplAppsecret, privateKey, returnMessage);
                JSONObject jo = new JSONObject(response.get("0"));
                log.debug("西银惠付下单返回的信息为 " + jo);
                // 0000—请求通讯成功
                if ("0000".equals(jo.getString("H_RSP_CODE"))) {

                    result.put("f_out_trade_no", outTradeNo);
                    result.put("openol_appid", openplAppId);
                    result.put("f_filiale", filiale);
                    result.put("flag", "MicroPayTongChuan");

                    // 保存中间表对象
                    JSONObject saveOrder = new JSONObject();
                    // 交易状态 tradeState
                    String tradeState = jo.getString("TRADE_STATE");
                    if ("NOTPAY".equals(tradeState) || "USERPAYING".equals(tradeState)) {
                        // NOTPAY -未支付, USERPAYING - 等待客户验证密码
                        saveOrder.put("f_order_state", "待查询");
                        result.put("result_msg", "支付结果未知");
                    } else if ("SUCCESS".equals(tradeState)) {
                        // 支付成功
                        saveOrder.put("f_order_state", "已支付");
                        // 支付完成时间
                        saveOrder.put("f_time_end", jo.getString("TIME_END"));
                        result.put("result_msg", "支付确认成功");
                    } else {
                        throw new RuntimeException("付款码下单失败,银行返回信息:" + jo);
                    }
                    // 支付渠道 PAY_PASS
                    String orderType = jo.optString("PAY_PASS");
                    // 000 六码合一 100 西安银行 001 支付宝 010 微信 002 翼支付 003 银联支付 004 京东支付 005 QQ 钱包支付 POS 银行卡支付 101 智能 POS 刷卡支付
                    String payPass = "";
                    switch (orderType) {
                        case "000":
                            payPass = "六码合一";
                            break;
                        case "100":
                            payPass = "西安银行";
                            break;
                        case "001":
                            payPass = "支付宝";
                            break;
                        case "010":
                            payPass = "微信";
                            break;
                        case "002":
                            payPass = "翼支付";
                            break;
                        case "003":
                            payPass = "银联支付";
                            break;
                        case "004":
                            payPass = "京东支付";
                            break;
                        case "005":
                            payPass = "QQ钱包支付";
                            break;
                        case "POS":
                            payPass = "银行卡支付";
                            break;
                        case "101":
                            payPass = "智能POS刷卡支付";
                            break;
                        default:
                            payPass = "未知";
                    }
                    result.put("qrcodetype", payPass);

                    // 保存下单信息到中间表
                    JSONObject param = new JSONObject(attach);
                    String userFilesId = param.optString("f_userfiles_id", "");
                    // 商户号
                    saveOrder.put("f_mch_id", jo.optString("MCHNT_NO"));
                    saveOrder.put("f_out_trade_no", outTradeNo);
                    saveOrder.put("f_transaction_id", jo.optString("TRANSACTION_ID"));
                    saveOrder.put("f_attach", attach);
                    saveOrder.put("f_order_type", "燃气收费");
                    saveOrder.put("flag", "MicroPayTongChuan");
                    saveOrder.put("f_trade_type", "MICROPAY");
                    saveOrder.put("f_bank_type", payPass);
                    saveOrder.put("f_filiale", filiale);
                    saveOrder.put("f_userfiles_id", userFilesId);
                    saveOrder.put("f_openid", jo.optString("OPENID"));
                    saveOrder.put("f_cash_fee", jo.optString("CASH_FEE"));
                    saveOrder.put("f_total_fee", jo.optString("TOTAL_FEE"));
                    // 付款码支付,下单时就把时间给存进去
                    saveOrder.put("f_time_end", DateUtil.format(DateUtil.date(), DatePattern.PURE_DATETIME_PATTERN));
                    // 保存分公司id
                    JSONObject clientConfig = Config.getClientConfig(filiale);
                    saveOrder.put("f_orgid", clientConfig.get("orgStr"));
                    logicServer.run("savewxreturnxml", saveOrder);
                }
            }
        } catch (Exception e) {
            log.debug("西银惠付款码下单异常错误", e);
            result.put("result_msg", "支付确认失败");
            result.put("err_msg", e.getMessage());
        }
        return result.toString();
    }

    /**
     * 铜川被扫支付订单状态查询
     * @param value
     * @return
     */
    @Override
    public String orderStatus(String value) {
        JSONObject result = new JSONObject();
        try {
            JSONObject jsonObject = new JSONObject(value);
            JSONObject wxConfig = Config.getConfig(jsonObject.getString("f_filiale"));
            // 私钥
            String privateKey = wxConfig.getString("privateKey");
            // 用户密钥
            String openplAppsecret = wxConfig.getString("openpl_appsecret");
            // APP_ID
            String openplAppId = jsonObject.getString("openol_appid");
            // 订单号
            String outTradeNo = jsonObject.getString("out_trade_no");
            // 签名
            JSONObject jsonString = new JSONObject();
            jsonString.put("OPENPL_APPID", openplAppId);
            jsonString.put("OUT_TRADE_NO", outTradeNo);
            // 加密并格式化发送消息
            String postMessage = FormatUtils.encryptMessageByPrivateKey(openplAppsecret, privateKey, String.valueOf(jsonString));
            // 查询地址
            String microqueryUrl = wxConfig.getString("MicroqueryUrl");
            log.debug("西银惠付付款码查询地址: {},查询参数: {}", microqueryUrl, jsonString);
            byte[] data = null;
            try {
                data = HttpNetProvider.doPostBytes(microqueryUrl, postMessage, "application/json");
            } catch (Exception e) {
                log.debug("请求响应错误");
                result.put("result_msg", "请求响应错误");
                result.put("err_msg", e.getMessage());
                throw new RuntimeException("付款码下单请求西银惠付未正常响应");
            }
            if (data != null) {
                String returnMessage = new String(data, StandardCharsets.UTF_8);
                Map<String, String> response = FormatUtils.handleReturnMessage(openplAppsecret, privateKey, returnMessage);
                JSONObject jo = new JSONObject(response.get("0"));
                log.debug("西银惠付查询返回的信息为 " + jo);
                // 0000—请求通讯成功
                if ("0000".equals(jo.getString("H_RSP_CODE"))) {
                    // 交易状态
                    // SUCCESS—支付成功 REFUND—已退款 NOTPAY—未支付 CLOSED—已关闭 USERPAYING--用户支付中 PAYERROR--支付失败 PARTREFUND-部分退款
                    String tradeState = jo.getString("TRADE_STATE");
                    if ("NOTPAY".equals(tradeState) || "USERPAYING".equals(tradeState)) {
                        // NOTPAY -未支付, USERPAYING - 等待客户验证密码、用户支付中
                        result.put("result_msg", "支付结果未知");
                    } else if ("SUCCESS".equals(tradeState)) {
                        // 支付成功
                        result.put("result_msg", "支付确认成功");
                        JSONArray query = this.sqlServer.query("select id from t_weixinreturnxml where f_out_trade_no = '" + outTradeNo + "'");
                        // 保存中间表对象
                        JSONObject saveOrder = new JSONObject();
                        if (query.length() > 0) {
                            // 支付渠道 PAY_PASS
                            String orderType = jo.optString("PAY_PASS");
                            // 000 六码合一 100 西安银行 001 支付宝 010 微信 002 翼支付 003 银联支付 004 京东支付 005 QQ 钱包支付 POS 银行卡支付 101 智能 POS 刷卡支付
                            String payPass = "";
                            switch (orderType) {
                                case "000":
                                    payPass = "六码合一";
                                    break;
                                case "100":
                                    payPass = "西安银行";
                                    break;
                                case "001":
                                    payPass = "支付宝";
                                    break;
                                case "010":
                                    payPass = "微信";
                                    break;
                                case "002":
                                    payPass = "翼支付";
                                    break;
                                case "003":
                                    payPass = "银联支付";
                                    break;
                                case "004":
                                    payPass = "京东支付";
                                    break;
                                case "005":
                                    payPass = "QQ钱包支付";
                                    break;
                                case "POS":
                                    payPass = "银行卡支付";
                                    break;
                                case "101":
                                    payPass = "智能POS刷卡支付";
                                    break;
                                default:
                                    payPass = "未知";
                            }
                            // 商户号
                            saveOrder.put("f_mch_id", jo.optString("MCHNT_NO"));
                            saveOrder.put("f_transaction_id", jo.optString("TRANSACTION_ID"));
                            saveOrder.put("f_bank_type", payPass);
                            saveOrder.put("f_openid", jo.optString("OPENID"));
                            saveOrder.put("f_cash_fee", jo.optString("CASH_FEE"));
                            saveOrder.put("f_total_fee", jo.optString("TOTAL_FEE"));
                            saveOrder.put("f_time_end", DateUtil.format(DateUtil.date(), DatePattern.PURE_DATETIME_PATTERN));
                            saveOrder.put("id", query.getJSONObject(0).get("id"));
                            logicServer.run("savewxreturnxml", saveOrder);
                        }
                    } else {
                        throw new RuntimeException("付款码下单失败,银行返回信息:" + jo);
                    }
                } else {
                    result.put("result_msg", "查询通讯失败");
                    result.put("err_msg", jo.getString("H_RSP_MSG"));
                }
            }
        } catch (Exception e) {
            log.debug("西银惠付付款码查询异常错误:", e);
            result.put("result_msg", "支付确认失败");
            result.put("err_msg", e.getMessage());
        }
        return result.toString();
    }
}
