package com.aote.pay.icbc.weinan;

import java.io.UnsupportedEncodingException;
import java.security.PublicKey;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;
import java.util.Map.Entry;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class SDKUtil {
	public static Logger logger = LoggerFactory.getLogger(SDKUtil.class);

	/**
	 * 生成签名值(SHA1摘要算法),供商户调用
	 *
	 * @param data
	 *            待签名数据Map键值对形式
	 * @param encoding
	 *            编码
	 * @return 签名是否成功
	 */
	public static boolean sign(Map<String, String> data, String encoding,
			CertUtil CertUtil) {
		logger.debug("签名处理开始.");
		if (isEmpty(encoding)) {
			encoding = "UTF-8";
		}
		// 设置签名证书序列号
		data.put( SDKConstants.param_certId, CertUtil.getSignCertId());

		// 将Map信息转换成key1=value1&key2=value2的形式
		String stringData = coverMap2String(data);

		logger.debug("报文签名之前的字符串(不含signature域)=[" + stringData + "]");
		/**
		 * 签名\base64编码
		 */
		byte[] byteSign = null;
		String stringSign = null;
		try {
			// 通过SHA1进行摘要并转16进制
			byte[] signDigest = SecureUtil.sha1X16(stringData, encoding);
			logger.debug("SHA1->16进制转换后的摘要=[" + new String(signDigest) + "]");
			byteSign = SecureUtil.base64Encode(SecureUtil.signBySoft(
					CertUtil.getSignCertPrivateKey(), signDigest));
			stringSign = new String(byteSign);
			logger.debug("报文签名之后的字符串=[" + stringSign + "]");
			// 设置签名域值
			data.put(SDKConstants.param_signature, stringSign);
			logger.debug("签名处理结束.");
			return true;
		} catch (Exception e) {
			logger.debug("签名异常,{}", e);
			// LogUtil.writeErrorLog("签名异常", e);
			return false;
		}
	}

	/**
	 * 通过传入的证书绝对路径和证书密码读取签名证书进行签名并返回签名值<br>
	 * 此为信用卡应用提供的特殊化处理方法
	 *
	 * @param data
	 *            待签名数据Map键值对形式
	 * @param encoding
	 *            编码
	 * @param certPath
	 *            证书绝对路径
	 * @param certPwd
	 *            证书密码
	 * @return 签名值
	 */
	public static boolean signByCertInfo(Map<String, String> data,
			String encoding, String certPath, String certPwd, CertUtil CertUtil) {
		logger.debug("签名处理开始.");
		if (isEmpty(encoding)) {
			encoding = "UTF-8";
		}
		if (isEmpty(certPath) || isEmpty(certPwd)) {
			logger.debug("传入参数不合法,签名失败");
			return false;
		}
		// 设置签名证书序列号
		data.put(SDKConstants.param_certId,
				CertUtil.getCertIdByCertPath(certPath, certPwd, "PKCS12"));
		// 将Map信息转换成key1=value1&key2=value2的形式
		String stringData = coverMap2String(data);
		logger.debug("报文签名之前的字符串(不含signature域)=[" + stringData + "]");
		/**
		 * 签名\base64编码
		 */
		byte[] byteSign = null;
		String stringSign = null;
		try {
			byte[] signDigest = SecureUtil.sha1X16(stringData, encoding);
			logger.debug("SHA1->16进制转换后的摘要=[" + new String(signDigest) + "]");
			byteSign = SecureUtil.base64Encode(SecureUtil.signBySoft(
					CertUtil.getSignCertPrivateKey(certPath, certPwd),
					signDigest));
			stringSign = new String(byteSign);
			logger.debug("报文签名之后的字符串=[" + stringSign + "]");
			// 设置签名域值
			data.put(SDKConstants.param_signature, stringSign);
			logger.debug("签名处理结束.");
			return true;
		} catch (Exception e) {
			logger.debug("签名异常,{}", e);
			// LogUtil.writeErrorLog("签名异常", e);
			return false;
		}
	}


	/**
	 * 验证签名(SHA-1摘要算法)
	 *
	 * @param resData
	 *            返回报文数据
	 * @param encoding
	 *            编码格式
	 * @return
	 */
	public static boolean validate(Map<String, String> resData,
			String encoding, String url) {
		logger.debug("验签处理开始.");
		if (isEmpty(encoding)) {
			encoding = "UTF-8";
		}
		try {
			String stringSign = resData.get(SDKConstants.param_signature);
			logger.debug("返回报文中signature=[" + stringSign + "]");
			SDKConfig paySDKConfig = new SDKConfig();
			CertUtil payCertUtil = new CertUtil(paySDKConfig, url);
			PublicKey pkey = payCertUtil.getValidateKey(url);
			resData.put(SDKConstants.param_certId,
					payCertUtil.getAttestionCertId(url));
			// 从返回报文中获取certId ，然后去证书静态Map中查询对应验签证书对象
			String certId = resData.get(SDKConstants.param_certId);
			logger.debug("返回报文中certId=[" + certId + "]");
			// 将Map信息转换成key1=value1&key2=value2的形式
			String stringData = coverMap2String(resData);
			logger.debug("返回报文中(不含signature域)的stringData=[" + stringData + "]");
			// 验证签名需要用银联发给商户的公钥证书.
			return SecureUtil.validateSignBySoft(pkey,
					SecureUtil.base64Decode(stringSign.getBytes(encoding)),
					SecureUtil.sha1X16(stringData, encoding));
		} catch (UnsupportedEncodingException e) {
			logger.debug("validate exception1===>{}", e);
			LogUtil.writeErrorLog(e.getMessage(), e);
		} catch (Exception e) {
			logger.debug("validate exception2===>{}", e);
			LogUtil.writeErrorLog(e.getMessage(), e);
		}
		return false;
	}

	/**
	 * 将Map中的数据转换成key1=value1&key2=value2的形式 不包含签名域signature
	 *
	 * @param data
	 *            待拼接的Map数据
	 * @return 拼接好后的字符串
	 */
	public static String coverMap2String(Map<String, String> data) {
		TreeMap<String, String> tree = new TreeMap<String, String>();
		Iterator<Entry<String, String>> it = data.entrySet().iterator();
		while (it.hasNext()) {
			Entry<String, String> en = it.next();
			if (SDKConstants.param_signature.equals(en.getKey().trim())) {
				continue;
			}
			tree.put(en.getKey(), en.getValue());
		}
		it = tree.entrySet().iterator();
		StringBuffer sf = new StringBuffer();
		while (it.hasNext()) {
			Entry<String, String> en = it.next();
			sf.append(en.getKey() + SDKConstants.EQUAL + en.getValue()
					+ SDKConstants.AMPERSAND);
		}
		return sf.substring(0, sf.length() - 1);
	}
	/**
	 * 判断字符串是否为NULL或空
	 *
	 * @param s
	 *            待判断的字符串数据
	 * @return 判断结果 true-是 false-否
	 */
	public static boolean isEmpty(String s) {
		return null == s || "".equals(s.trim());
	}
}
