package com.aote.pay.icbc.weinan;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Security;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;


public class CertUtil {
	/** 证书容器. */
	private  KeyStore keyStore = null;
	/** 验证签名证书. */
	private  X509Certificate validateCert = null;

	/** 根据传入证书文件路径和密码读取指定的证书容器. */
	private  KeyStore certKeyStore = null;

	private Map <String,String> attCertMap = new HashMap<String, String>();

	/**
	 * 初始化证书容器的方法
	 */
	SDKConfig config;
	/**
	 * 初始化所有证书.
	 * 验签证书路径url(商户号)为空使用签名证书          不为空使用验签证书
	 */
	public  CertUtil(SDKConfig config,String url) {
		this.config=config;
		if(SDKUtil.isEmpty(url)) {
			initSignCert(config);
		}else {
			initValidateCertFromDir(url);
		}
	}

	/**
	 * 加载签名证书
	 */
	public  void initSignCert(SDKConfig config) {
		System.out.println("加载签名证书开始");
		if (null != keyStore) {
			keyStore = null;
		}
		keyStore = getKeyInfo(config.getSignCertPath(),
				config.getSignCertPwd(), config
						.getSignCertType());
		// 打印证书加载信息,供测试阶段调试
		System.out.println("[" + config.getSignCertPath()
				+ "][serialNumber=" + getSignCertId() + "]");
		System.out.println("加载签名证书结束");
	}

	/**
	 * 根据传入的证书文件路径和证书密码加载指定的签名证书
	 */
	public  void initSignCert(String certFilePath, String certPwd) {

		System.out.println("加载证书文件[" + certFilePath + "]和证书密码[" + certPwd
				+ "]的签名证书开始.");
		File files = new File(certFilePath);
		if (!files.exists()) {
			System.out.println("证书文件不存在,初始化签名证书失败.");
			return;
		}
		if (null != certKeyStore) {
			certKeyStore = null;
		}
		certKeyStore = getKeyInfo(certFilePath, certPwd, "PKCS12");
		System.out.println("加载证书文件[" + certFilePath + "]和证书密码[" + certPwd
				+ "]的签名证书结束.");
	}

	/**
	 * 从指定目录下加载验证签名证书
	 *
	 */
	public X509Certificate initValidateCertFromDir(String filePath) {
		System.out.println("从目录中加载验证签名证书开始.");
//		certMap.clear();
		if (null == filePath || "".equals(filePath)) {
			System.out.println("验证签名证书路径配置为空.");
			return null;
		}

		CertificateFactory cf = null;
		FileInputStream in = null;

		try {
			cf = CertificateFactory.getInstance("X.509");
			File file=new File(filePath);
				in = new FileInputStream(file.getAbsolutePath());
				validateCert = (X509Certificate) cf.generateCertificate(in);
				System.out.println("[" + file.getAbsolutePath()
						+ "][serialNumber="
						+ validateCert.getSerialNumber().toString() + "]");
				System.out.println("从目录中加载验证签名证书结束.");
				attCertMap.put(filePath, validateCert.getSerialNumber().toString());
				return validateCert;
		} catch (CertificateException e) {
			LogUtil.writeErrorLog("验证签名证书加载失败", e);
			return null;
		} catch (FileNotFoundException e) {
			LogUtil.writeErrorLog("验证签名证书加载失败,证书文件不存在", e);
			return null;
		} finally {
			if (null != in) {
				try {
					in.close();
				} catch (IOException e) {
					LogUtil.writeErrorLog(e.toString());
				}
			}
		}
	}

	/**
	 * 获取签名证书私钥
	 *
	 * @return
	 */
	public  PrivateKey getSignCertPrivateKey() {
		try {
			Enumeration<String> aliasenum = keyStore.aliases();
			String keyAlias = null;
			if (aliasenum.hasMoreElements()) {
				keyAlias = aliasenum.nextElement();
			}
			PrivateKey privateKey = (PrivateKey) keyStore.getKey(keyAlias,
					config.getSignCertPwd().toCharArray());
			return privateKey;
		} catch (Exception e) {
			LogUtil.writeErrorLog("获取签名证书的私钥失败", e);
			return null;
		}
	}

	/**
	 * 通过传入证书绝对路径和证书密码获取所对应的签名证书私钥
	 *
	 * @param certPath
	 *            证书绝对路径
	 * @param certPwd
	 *            证书密码
	 * @return 证书私钥
	 */
	public  PrivateKey getSignCertPrivateKey(String certPath,
			String certPwd) {
		// 初始化指定certPath和certPwd的签名证书容器
		initSignCert(certPath, certPwd);
		try {
			Enumeration<String> aliasenum = certKeyStore.aliases();
			String keyAlias = null;
			if (aliasenum.hasMoreElements()) {
				keyAlias = aliasenum.nextElement();
			}
			PrivateKey privateKey = (PrivateKey) certKeyStore.getKey(keyAlias,
					certPwd.toCharArray());
			return privateKey;
		} catch (Exception e) {
			LogUtil.writeErrorLog("获取[" + certPath + "]的签名证书的私钥失败", e);
			return null;
		}
	}

	/**
	 * 验证签名证书
	 *
	 * @return 验证签名证书的公钥
	 */
	public  PublicKey getValidateKey() {
		try {
			if (null == validateCert) {
				return null;
			}
			return validateCert.getPublicKey();
		} catch (Exception e) {
			LogUtil.writeErrorLog("获取验证签名证书失败", e);
			return null;
		}
	}

	/**
	 * 通过certId获取证书Map中对应证书的公钥
	 *
	 * @param certId
	 *            证书物理序号
	 * @return 通过证书编号获取到的公钥
	 */
	public  PublicKey getValidateKey(String mchnt_no) {
			return initValidateCertFromDir(mchnt_no).getPublicKey();
	}

	/**
	 * 获取签名证书中的证书序列号
	 *
	 * @return 证书的物理编号
	 */
	public  String getSignCertId() {
		try {
			Enumeration<String> aliasenum = keyStore.aliases();
			String keyAlias = null;
			if (aliasenum.hasMoreElements()) {
				keyAlias = aliasenum.nextElement();
			}
			X509Certificate cert = (X509Certificate) keyStore
					.getCertificate(keyAlias);
			return cert.getSerialNumber().toString();
		} catch (Exception e) {
			LogUtil.writeErrorLog("获取签名证书的序列号失败", e);
			if (null == keyStore) {
				LogUtil.writeErrorLog("keyStore实例化失败,当前为NULL");
			}
			return "";
		}
	}

	public String getAttestionCertId(String mchnt_no ) {
		return attCertMap.get(mchnt_no);
	}
	/**
	 * 获取签名证书公钥对象
	 *
	 * @return
	 */
	public  PublicKey getSignPublicKey() {
		try {
			Enumeration<String> aliasenum = keyStore.aliases();
			String keyAlias = null;
			if (aliasenum.hasMoreElements()) // we are readin just one
			// certificate.
			{
				keyAlias = (String) aliasenum.nextElement();
			}

			Certificate cert = keyStore.getCertificate(keyAlias);
			PublicKey pubkey = cert.getPublicKey();
			return pubkey;
		} catch (Exception e) {
			LogUtil.writeErrorLog(e.toString());
			return null;
		}
	}

	/**
	 * 将证书文件读取为证书存储对象
	 *
	 * @param pfxkeyfile
	 *            证书文件名
	 * @param keypwd
	 *            证书密码
	 * @param type
	 *            证书类型
	 * @return 证书对象
	 */
	public  KeyStore getKeyInfo(String pfxkeyfile, String keypwd,
			String type) {
		try {
			System.out.println("KeyStore Loading Start...");
			KeyStore ks = null;
			if ("JKS".equals(type)) {
				ks = KeyStore.getInstance(type);
			} else if ("PKCS12".equals(type)) {
				Security
						.addProvider(new org.bouncycastle.jce.provider.BouncyCastleProvider());
				// ks = KeyStore.getInstance(type, "BC");
				ks = KeyStore.getInstance(type);
			}
			System.out.println("传入的私钥证书路径为=>[" + pfxkeyfile + "],密码=[" + keypwd
					+ "]");
			FileInputStream fis = new FileInputStream(pfxkeyfile);
			char[] nPassword = null;
			nPassword = null == keypwd || "".equals(keypwd.trim()) ? null
					: keypwd.toCharArray();
			if (null != ks){
				ks.load(fis, nPassword);
			}
			fis.close();
			System.out.println("KeyStore Loading End...");
			return ks;
		} catch (Exception e) {
			if (Security.getProvider("BC") == null) {
				System.out.println("BC Provider not installed.");
			}
			LogUtil.writeErrorLog("读取私钥证书失败", e);
			if ((e instanceof KeyStoreException) && "PKCS12".equals(type)) {
				Security.removeProvider("BC");
			}
			return null;
		}
	}

	/**
	 * 通过传入私钥证书路径，获取证书ID
	 *
	 * @param path
	 * @param pwd
	 * @param certTp
	 * @return
	 */
	public  String getCertIdByCertPath(String path, String pwd,
			String certTp) {
		KeyStore ks = getKeyInfo(path, pwd, certTp);
		if (null == ks) {
			return "";
		}
		try {
			Enumeration<String> aliasenum = ks.aliases();
			String keyAlias = null;
			if (aliasenum.hasMoreElements()) {
				keyAlias = aliasenum.nextElement();
			}
			X509Certificate cert = (X509Certificate) ks
					.getCertificate(keyAlias);
			return cert.getSerialNumber().toString();
		} catch (Exception e) {
			LogUtil.writeErrorLog("获取签名证书的序列号失败", e);
			return "";
		}
	}
}
