package com.aote.webmeter.timer;

import com.af.plugins.CommonTools;
import com.af.plugins.JsonTools;
import com.af.plugins.RestTools;
import com.aote.module.ModuleMapper;
import com.aote.rs.LogicService;
import com.aote.rs.SqlService;
import com.aote.sql.SqlMapper;
import com.aote.webmeter.enums.RestCodeEnum;
import com.aote.webmeter.enums.SaleVersionEnum;
import com.aote.webmeter.enums.WebmeterPropertiesEnum;
import com.aote.webmeter.exception.WebmeterException;
import com.aote.webmeter.tools.CompatTools;
import com.aote.webmeter.tools.WebMeterInfo;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 物联网表Timer
 * *****
 * 重要：该版本定时器为正式版本，为目前稳定的定时业务处理机制
 * *****
 * 保存指令业务配置：saveBusiness.json
 * 发送指令业务配置：sendBusiness.json
 * 发送二级指令业务配置：sendBusiness2.json
 * 发送三级指令业务配置：sendBusiness3.json
 * <p>
 * 注：发送指令中的一级,二级,三级没有实现上的区别，只用作定时业务时，区分不同类型的业务
 * </p>
 *
 * @author Mrriver
 * @date 2020/4/23
 */
@Component
public class WebMeterTimer {
    /**
     * 日志对象
     */
    private static final Logger LOGGER = Logger.getLogger(WebMeterTimer.class);
    /**
     * 指令业务列表
     */
    private static final Map<String, JSONArray> SAVE_BUSINESS_LIST = new HashMap<>(3);
    private static final String SAVE_BUSINESS_LIST_PATH = "saveBusiness.json";
    /**
     * 发送指令业务列表
     */
    private static final Map<String, JSONArray> SEND_BUSINESS_LIST = new HashMap<>(3);
    private static final String SEND_BUSINESS_LIST_PATH = "sendBusiness.json";
    /**
     * 发送二级业务指令列表
     */
    private static final Map<String, JSONArray> SEND_BUSINESS_LIST_2 = new HashMap<>(3);
    private static final String SEND_BUSINESS_LIST_PATH_2 = "sendBusiness2.json";
    /**
     * 发送三级业务指令列表
     */
    private static final Map<String, JSONArray> SEND_BUSINESS_LIST_3 = new HashMap<>(3);
    private static final String SEND_BUSINESS_LIST_PATH_3 = "sendBusiness3.json";
    /**
     * SQL查询支持
     */
    @Autowired
    private SqlService sqlService;
    /**
     * Logic执行支持
     */
    @Autowired
    private LogicService logicService;


    public WebMeterTimer() {
        load();
    }

    /**
     * 读取业务配置文件
     */
    public static void load() {
        // 获取所有引入的表具模块名
        Map<String, Map<String, String>> map = ModuleMapper.getMap();
        for (String moduleName : map.keySet()) {
            String modulePath = map.get(moduleName).get("path");
            if (modulePath == null) {
                modulePath = "";
            } else {
                modulePath = modulePath + "/";
            }
            //获取每个表具模块下的业务配置文件
            String saveBusinessListPath = modulePath + moduleName + "/" + SAVE_BUSINESS_LIST_PATH;
            String sendBusinessListPath = modulePath + moduleName + "/" + SEND_BUSINESS_LIST_PATH;
            String sendBusinessListPath2 = modulePath + moduleName + "/" + SEND_BUSINESS_LIST_PATH_2;
            String sendBusinessListPath3 = modulePath + moduleName + "/" + SEND_BUSINESS_LIST_PATH_3;
            if (WebMeterTimer.class.getClassLoader().getResourceAsStream(saveBusinessListPath) != null) {
                JSONArray business = JsonTools.readJsonArrayFile(saveBusinessListPath);
                SAVE_BUSINESS_LIST.put(moduleName, business);
            }
            if (WebMeterTimer.class.getClassLoader().getResourceAsStream(sendBusinessListPath) != null) {
                JSONArray business = JsonTools.readJsonArrayFile(sendBusinessListPath);
                SEND_BUSINESS_LIST.put(moduleName, business);
            }
            if (WebMeterTimer.class.getClassLoader().getResourceAsStream(sendBusinessListPath2) != null) {
                JSONArray business2 = JsonTools.readJsonArrayFile(sendBusinessListPath2);
                SEND_BUSINESS_LIST_2.put(moduleName, business2);
            }
            if (WebMeterTimer.class.getClassLoader().getResourceAsStream(sendBusinessListPath3) != null) {
                JSONArray business3 = JsonTools.readJsonArrayFile(sendBusinessListPath3);
                SEND_BUSINESS_LIST_3.put(moduleName, business3);
            }
        }
    }

    /**
     * 清理业务配置文件
     */
    public static void clear() {
        SAVE_BUSINESS_LIST.clear();
        SEND_BUSINESS_LIST.clear();
        SEND_BUSINESS_LIST_2.clear();
        SEND_BUSINESS_LIST_3.clear();
    }

    /**
     * 执行单个业务
     *
     * @param type 业务类型
     */
    public void doBusiness(String moduleName, String alias, String type, JSONObject business) {
        //业务名称
        String businessName = business.getString("name");
        //业务逻辑(Logic)名称
        String logicName = business.getString("logicName");
        //组织执行数据
        JSONObject data = business.getJSONObject("data");
        data.put("moduleName", moduleName);
        //开始执行业务
        LOGGER.info("======模块：【" + moduleName + "】开始" + type + "【" + businessName + "】指令业务流程======");
        //业务所需SQL
        JSONObject sqlData;
        //如果有SQL资源
        boolean isRunSql = business.has("sqlData");
        if (isRunSql) {
            sqlData = business.getJSONObject("sqlData");
            //获取SQL结果集
            JSONArray resultArray = getSqlResult(sqlData, alias);
            if (resultArray != null) {
                resultArray.forEach((item) -> {
                    //追加请求数据
                    data.put("object", item);
                    //执行Logic
                    runLogic(logicName, data);
                });
            }
        } else {
            //执行Logic
            runLogic(logicName, data);
        }
        LOGGER.info("======模块：【" + moduleName + "】" + type + "【" + businessName + "】指令业务流程结束======");
    }

    /**
     * 获取业务执行所需SQL的结果集
     *
     * @param sqlData SQL请求参数
     * @param alias   别名
     * @return 结果集
     */
    public JSONArray getSqlResult(JSONObject sqlData, String alias) {
        if (sqlData.has("isWeb")) {
            //请求地址
            String sqlUrl = WebMeterInfo.getString(sqlData.getString("sqlUrl"));
            //查询条件
            JSONObject sqlParams = sqlData.getJSONObject("sqlParams");
            //请求方式
            String restType = sqlData.getString("restType");
            //请求结果
            String result = "";
            try {
                if (restType.equals(RestCodeEnum.GET.getValue())) {
                    result = RestTools.get(sqlUrl, sqlParams);
                } else {
                    result = RestTools.post(sqlUrl, sqlParams);
                }
                return JsonTools.parseArray(result);
            } catch (JSONException e) {
                LOGGER.error("请求业务所需SQL的网络资源时，没有获取到预期值。返回值：" + result);
                return null;
            } catch (Exception e) {
                LOGGER.error("出现异常：", e);
                return null;
            }
        } else {
            // 获取营收版本
            SaleVersionEnum saleVersionEnum;
            try {
                saleVersionEnum = WebMeterInfo.getSaleNameEnum();
            } catch (WebmeterException e) {
                saleVersionEnum = null;
            }
            //自动版本推断
            boolean isAutoVersion = sqlData.has("isAutoVersion") && sqlData.getBoolean("isAutoVersion");
            //如果未启用自动版本推断，将寻找自定义配置的资源
            if (!isAutoVersion) {
                if (saleVersionEnum != null) {
                    // 获取所有营收版本的SQL
                    Set<String> regExSet = sqlData.keySet();
                    for (String versionRegEx : regExSet) {
                        // 编译正则表达式
                        Pattern pattern = Pattern.compile("^" + versionRegEx + "$", Pattern.CASE_INSENSITIVE);
                        // 寻找该营收版本的SQL
                        Matcher matcher = pattern.matcher(saleVersionEnum.getValue());
                        if (matcher.find()) {
                            sqlData = sqlData.getJSONObject(versionRegEx);
                            break;
                        }
                    }
                }
            }
            if (sqlData.has("sqlName")) {
                //SQL名称
                String sqlName = sqlData.getString("sqlName");
                //如果启用了自动版本推断，将根据预定义名称和营收版本获取资源映射名称
                if (isAutoVersion) {
                    if (sqlData.has("baseType")) {
                        sqlName = CompatTools.getResName(sqlName, sqlData.getInt("baseType"));
                    } else {
                        sqlName = CompatTools.getResName(sqlName);
                    }
                }
                //自动数据源版本推断
                boolean isAutoDataSource = sqlData.has("isAutoDataSource") && sqlData.getBoolean("isAutoDataSource");
                if (isAutoDataSource) {
                    sqlName = CompatTools.getDataSourceResName(sqlName);
                }
                if (SqlMapper.getSql(sqlName) == null) {
                    LOGGER.info("未找到" + WebMeterInfo.getSaleName() + "中名为【" + sqlName + "】的SQL映射文件");
                    return null;
                }
                //查询条件
                JSONObject sqlParams = sqlData.getJSONObject("sqlParams");
                if (!sqlParams.isEmpty()) {
                    //对查询SQL的参数进行格式化
                    for (String paramKey : sqlParams.keySet()) {
                        //格式化参数字符串
                        String value = CommonTools.parametricStr(sqlParams.getString(paramKey), WebMeterInfo.getMap());
                        sqlParams.put(paramKey, value);
                    }
                }

                //根据别名查询气表品牌ID，优化同步查询
                if (saleVersionEnum != null) {
                    switch (saleVersionEnum) {
                        case Vue_V1:
                        case Vue_V2:
                        case Vue_V3:
                            JSONArray gasBrandArray;
                            try {
                                gasBrandArray = new JSONArray(sqlService.txQuery("SELECT id FROM t_gasbrand WHERE f_alias = '" + alias + "'"));
                            } catch (Exception e) {
                                LOGGER.error("出现异常：", e);
                                return null;
                            }
                            String gasBrandIds;
                            if (gasBrandArray.length() > 0) {
                                gasBrandIds = CommonTools.union(gasBrandArray, "id");
                            } else {
                                gasBrandIds = "'-1'";
                            }
                            //查询条件注入表具品牌ID数组参数
                            sqlParams.put("gasBrandIds", gasBrandIds);
                            break;
                        default:
                            break;
                    }
                }
                //查询条件注入表具品牌别名参数
                sqlParams.put("f_meter_brand", alias);
                //查询条件注入表档案主键字段名参数
                sqlParams.put("userIdColumn", CompatTools.getUserFilesId());
                //查询数量
                int pageSize = 2000;
                //如果有查询条数限制
                boolean isHasPageSize = sqlData.has("pageSize");
                if (isHasPageSize) {
                    pageSize = sqlData.getInt("pageSize");
                    if (pageSize == 0) {
                        pageSize = 9999999;
                    }
                }
                try {
                    JSONObject params = new JSONObject();
                    params.put("data", sqlParams);
                    return new JSONArray(sqlService.txExecute(sqlName, 1, pageSize, params.toString()));
                } catch (Exception e) {
                    LOGGER.error("出现异常：", e);
                    return null;
                }
            } else {
                return null;
            }
        }
    }

    private void runLogic(String logicName, JSONObject data) {
        try {
            logicService.xtSave(logicName, data.toString());
        } catch (Exception e) {
            LOGGER.error("【" + logicName + "】出现异常：", e);
        }
    }

    /**
     * 执行保存指令业务
     */
    public void doSaveBusiness() {
        //遍历所有物联网表模块
        for (Map.Entry<String, JSONArray> entry : SAVE_BUSINESS_LIST.entrySet()) {
            //物联网表模块名
            String moduleName = entry.getKey();
            WebMeterInfo.setModuleName(moduleName);
            //获取表具品牌别名
            String alias = WebMeterInfo.getString(WebmeterPropertiesEnum.METER_ALIAS);
            entry.getValue().forEach((item) -> {
                //执行业务
                doBusiness(moduleName, alias, "保存", (JSONObject) item);
            });
        }
    }

    /**
     * 执行发送指令业务
     */
    public void doSendBusiness() {
        //遍历所有物联网表模块
        for (Map.Entry<String, JSONArray> entry : SEND_BUSINESS_LIST.entrySet()) {
            //物联网表模块名
            String moduleName = entry.getKey();
            WebMeterInfo.setModuleName(moduleName);
            //获取表具品牌别名
            String alias = WebMeterInfo.getString(WebmeterPropertiesEnum.METER_ALIAS);
            entry.getValue().forEach((item) -> {
                //执行业务
                doBusiness(moduleName, alias, "发送", (JSONObject) item);
            });
        }
    }

    /**
     * 执行发送指令二级业务
     */
    public void doSendBusiness2() {
        //遍历所有物联网表模块
        for (Map.Entry<String, JSONArray> entry : SEND_BUSINESS_LIST_2.entrySet()) {
            //物联网表模块名
            String moduleName = entry.getKey();
            WebMeterInfo.setModuleName(moduleName);
            //获取表具品牌别名
            String alias = WebMeterInfo.getString(WebmeterPropertiesEnum.METER_ALIAS);
            entry.getValue().forEach((item) -> {
                //执行业务
                doBusiness(moduleName, alias, "发送(二级)", (JSONObject) item);
            });
        }
    }

    /**
     * 执行发送指令三级业务
     */
    public void doSendBusiness3() {
        //遍历所有物联网表模块
        for (Map.Entry<String, JSONArray> entry : SEND_BUSINESS_LIST_3.entrySet()) {
            //物联网表模块名
            String moduleName = entry.getKey();
            WebMeterInfo.setModuleName(moduleName);
            //获取表具品牌别名
            String alias = WebMeterInfo.getString(WebmeterPropertiesEnum.METER_ALIAS);
            entry.getValue().forEach((item) -> {
                //执行业务
                doBusiness(moduleName, alias, "发送(三级)", (JSONObject) item);
            });
        }
    }
}
