package com.aote.webmeter.server.tcp;

import com.af.plugins.ConvertTools;
import com.aote.logic.LogicServer;
import com.aote.webmeter.enums.MsgTypeEnum;
import com.aote.webmeter.tools.tcp.TcpTools;
import org.apache.log4j.Logger;
import org.json.JSONObject;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.util.Arrays;
import java.util.Iterator;

/**
 * TCP服务
 *
 * @author Mrriver
 */
public class TcpServer extends Thread {
    private static final Logger log = Logger.getLogger(TcpServer.class);
    /**
     * TCP发送消息类型
     */
    public final MsgTypeEnum msgSendType;
    /**
     * TCP服务名称
     */
    private final String name;
    /**
     * TCP端口
     */
    private final int port;
    /**
     * logic对象
     */
    private final LogicServer service;
    /**
     * logic名称
     */
    private final String logicName;
    /**
     * TCP接收消息类型
     */
    private final MsgTypeEnum msgReceiveType;
    /**
     * 是否需要通道信息
     */
    private final boolean isHasChannelData;
    /**
     * 返回通道信息的Logic名称
     */
    private final String getChannelDataName;
    private Selector selector;

    public TcpServer(String name, String logicName, int port, LogicServer service,
                     MsgTypeEnum msgReceiveType, MsgTypeEnum msgSendType,
                     boolean isHasChannelData, String getChannelDataName) {
        this.name = name;
        this.logicName = logicName;
        this.port = port;
        this.service = service;
        this.msgReceiveType = msgReceiveType;
        this.msgSendType = msgSendType;
        this.isHasChannelData = isHasChannelData;
        this.getChannelDataName = getChannelDataName;
    }

    /**
     * 获得一个ServerSocket通道，并对该通道做一些初始化的工作
     */
    private void initServer() {
        try {
            // 获得一个通道管理器
            this.selector = Selector.open();
            // 打开通道
            ServerSocketChannel serverChannel = ServerSocketChannel.open();
            // 设置通道为非阻塞
            serverChannel.configureBlocking(false);
            // 将该通道对应的ServerSocket绑定ip和端口
            InetSocketAddress address = new InetSocketAddress(port);
            serverChannel.bind(address);
            // 注册事件
            serverChannel.register(selector, SelectionKey.OP_ACCEPT);
            // 监听事件
            listen();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void listen() throws Exception {
        log.debug("TCP服务端【" + name + "】的【" + port + "】端口启动成功！");
        // 轮询访问selector
        while (selector.select() > 0) {
            //当注册的事件到达时，方法返回；否则,该方法会一直阻塞
            selector.select();
            //获得selector中选中的项的迭代器，选中的项为注册的事件
            Iterator<SelectionKey> ite = this.selector.selectedKeys().iterator();
            while (ite.hasNext()) {
                SelectionKey key = ite.next();
                // 删除已选的key,以防重复处理
                ite.remove();
                if (key.isAcceptable()) {
                    ServerSocketChannel server = (ServerSocketChannel) key.channel();
                    SocketChannel sc = server.accept();
                    if (sc == null) {
                        continue;
                    }
                    sc.configureBlocking(false);
                    sc.register(selector, SelectionKey.OP_READ);
                    log.debug(sc.getRemoteAddress() + " 发起连接");
                } else if (key.isReadable()) {
                    // 获得了可读的事件
                    read(key);
                }
            }
        }
    }

    private void read(SelectionKey key) throws IOException {
        // 服务器可读取消息:得到事件发生的Socket通道
        SocketChannel channel = (SocketChannel) key.channel();
        // 设置成非阻塞
        channel.configureBlocking(false);
        // 创建读取的缓冲区
        ByteBuffer buffer = ByteBuffer.allocate(1024);
        // 客户端地址
        String address = channel.getRemoteAddress().toString();
        // 缓存通道
        TcpTools.setChannelMapValue(
                address,
                new TcpTools.ChannelData(msgSendType, channel));
        TcpTools.setChannelAddress(address);
        int recvCount;
        try {
            recvCount = channel.read(buffer);
        } catch (IOException e) {
            log.debug(address + "断开连接...");
            TcpTools.remove(address);
            log.debug("清除缓存...");
            buffer.clear();
            return;
        }
        if (recvCount > 0) {
            byte[] data = buffer.array();
            log.debug("服务端收到原始信息：" + Arrays.toString(data));
            String msg;
            switch (msgReceiveType) {
                case NORMAL:
                default:
                    msg = ConvertTools.bytesToStr(data);
                    break;
                case HEX:
                    msg = ConvertTools.byteToHexStr(data);
                    break;
            }
            // 去除空格，尾部0，转大写 (警告：该写法原用于处理一些地方由于缓冲区写入的数据尾部带有0导致解析异常的问题，
            // 但该写法得到的数据在莱德UDP表中出现了解析失败的情况)
            msg = msg.trim().replaceAll("0*$", "").toUpperCase();
            if (msgReceiveType == MsgTypeEnum.HEX && ((msg.length() & 0x01) != 0)) {
                msg += "0";
            }
            log.debug("服务端收到信息：" + msg);
            JSONObject params = new JSONObject();
            params.put("data", new JSONObject().put("value", msg));
            try {
                //如果需要通道信息
                if (isHasChannelData) {
                    JSONObject channelData = TcpTools.getChannelData(address, getChannelDataName, params, service);
                    params.put("channelData", channelData);
                }
                Object result = service.run(logicName, params);
                if (result != null && !result.equals("ok")) {
                    log.debug("服务端返回信息：" + result);
                    TcpTools.send(address, String.valueOf(result));
                }
            } catch (Exception e) {
                log.error("出现异常：", e);
                log.debug("服务端主动断开连接...");
                TcpTools.remove(address);
            }
        } else {
            log.debug("客户端请求断开连接...");
            TcpTools.remove(address);
        }
        log.debug("清除缓存...");
        buffer.clear();
    }

    @Override
    public void run() {
        initServer();
    }
}
