package com.aote.webmeter.tools.iot;

import com.af.plugins.HttpConnectionPoolUtil;
import com.aote.webmeter.enums.IOTBusinessTypeEnum;
import com.aote.webmeter.tools.WebMeterInfo;
import com.iotplatform.client.NorthApiException;
import com.iotplatform.constant.AuthConstant;
import com.iotplatform.constant.ExceptionEnum;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.log4j.Logger;
import org.json.JSONObject;

import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;
import java.io.IOException;
import java.io.InputStream;
import java.security.KeyStore;
import java.util.HashMap;
import java.util.Map;

/**
 * 电信IOT平台 应用接口鉴权
 * @author Mr.river
 */
public class AuthenticationTools {

	private static final Logger LOGGER = Logger.getLogger(AuthenticationTools.class);

	/**
	 * 鉴权有效时间
	 */
	private static final Map<String, Long> STAMP = new HashMap<>();
	/**
	 * 刷新鉴权有效时间
	 */
	private static final Map<String, Long> REFRESH_STAMP = new HashMap<>();
	/**
	 * 鉴权TOKEN
	 */
	private static final Map<String, String> AOD = new HashMap<>();
	/**
	 * 刷新鉴权Token
	 */
	private static final Map<String, String> REFRESH_TOKEN = new HashMap<>();
	/**
	 * 超时时间值
	 */
	private static final Integer TOKEN_TIMEOUT_VALUE = 1000 * 60 * 30;

	/**
	 * 发送请求的客户端单例
 	 */
	private static CloseableHttpClient httpClient;

	static {
		setHttpClient();
	}

	/**
	 * 获取HTTP客户端单例
	 */
	public static CloseableHttpClient getHttpClient(){
		if(httpClient == null) {
			setHttpClient();
		}
		return httpClient;
	}

	/**
	 * 设置HTTP客户端
	 */
	private static synchronized void setHttpClient(){
		try {
			httpClient = HttpConnectionPoolUtil.getHttpClient(initSslConfig());
		} catch (NorthApiException e) {
			e.printStackTrace();
		}
	}

	/**
	 * 获取TOKEN
	 * @return TOKEN值
	 */
	public static String getAuthToken() {
		String token = getToken();
		if (token != null) {
			return token;
		} else {
			try {
				return getWebAuthToken();
			} catch (IOException e) {
				e.printStackTrace();
				return null;
			}
		}
	}

	/**
	 * 发起获取TOKEN的请求
	 *
	 * @return TOKEN信息
	 * @throws IOException IO异常
	 */
	private static synchronized String getWebAuthToken() throws IOException {
		String token = getToken();
		if (token == null) {
			String appId = IotCommonTools.getIsAep() ? WebMeterInfo.getString("aepAPIKey") : WebMeterInfo.getString("telecomAPIKey");
			String secret = IotCommonTools.getIsAep() ? WebMeterInfo.getString("aepSecret") : WebMeterInfo.getString("telecomSecret");

			JSONObject content = new JSONObject();
			content.put("appId", appId);
			content.put("secret", secret);

			JSONObject result;
			if(getRefreshToken() == null) {
				LOGGER.info(appId + ":进行鉴权操作");
				result = IotCommonTools.post(IOTBusinessTypeEnum.AUTH, new HttpPost(), content, null);
			} else {
				LOGGER.info(appId + ":进行刷新鉴权操作");
				content.put("refreshToken",getRefreshToken());
				result = IotCommonTools.post(IOTBusinessTypeEnum.REFRESH_AUTH, new HttpPost(), content, null);
			}
			String refreshToken = result.getString("refreshToken");
			REFRESH_TOKEN.put(appId, refreshToken);
			REFRESH_STAMP.put(appId, System.currentTimeMillis());
			String newAccessToken = result.getString("accessToken");
			AOD.put(appId, newAccessToken);
			STAMP.put(appId, System.currentTimeMillis());
			return newAccessToken;
		}
		return token;
	}

	private static String getToken() {
		long current = System.currentTimeMillis();
		String appId = IotCommonTools.getIsAep() ? WebMeterInfo.getString("aepAPIKey") : WebMeterInfo.getString("telecomAPIKey");
		String accessToken = AOD.get(appId);
		Long stampValue = STAMP.get(appId);
		if (accessToken != null && stampValue != null && (current - stampValue) < TOKEN_TIMEOUT_VALUE) {
			LOGGER.info(appId + ":获取现有token");
			return accessToken;
		} else {
			return null;
		}
	}

	private static String getRefreshToken() {
		long current = System.currentTimeMillis();
		String appId = IotCommonTools.getIsAep() ? WebMeterInfo.getString("aepAPIKey") : WebMeterInfo.getString("telecomAPIKey");
		String refreshToken = REFRESH_TOKEN.get(appId);
		Long stampValue = REFRESH_STAMP.get(appId);
		if (refreshToken != null && stampValue != null && (current - stampValue) < TOKEN_TIMEOUT_VALUE) {
			LOGGER.info(appId + ":获取现有的刷新token");
			return refreshToken;
		} else {
			return null;
		}
	}

	/**
	 * 初始化SSL证书
	 *
	 * @return SSL连接工厂
	 * @throws NorthApiException IOT平台请求异常
	 */
	protected static SSLConnectionSocketFactory initSslConfig() throws NorthApiException {
		try (InputStream isTrustCa = SignalDeliveryTools.class.getResourceAsStream("/ca.jks");
			 InputStream isSelfCert = SignalDeliveryTools.class.getResourceAsStream("/outgoing.CertwithKey.pkcs12")) {
			KeyStore selfCert = KeyStore.getInstance("pkcs12");
			selfCert.load(isSelfCert, AuthConstant.SELFCERTPWD.toCharArray());
			KeyManagerFactory kmf = KeyManagerFactory.getInstance("sunx509");
			kmf.init(selfCert, AuthConstant.SELFCERTPWD.toCharArray());
			KeyStore caCert = KeyStore.getInstance("jks");
			caCert.load(isTrustCa, AuthConstant.TRUSTCAPWD.toCharArray());
			TrustManagerFactory tmf = TrustManagerFactory.getInstance("sunx509");
			tmf.init(caCert);
			SSLContext sc = SSLContext.getInstance("TLS");
			sc.init(kmf.getKeyManagers(), tmf.getTrustManagers(), null);
			return new SSLConnectionSocketFactory(sc, NoopHostnameVerifier.INSTANCE);
		} catch (Exception var10) {
			throw new NorthApiException(ExceptionEnum.CLIENT_SSL_CONFIG_ERROR, var10.getMessage());
		}
	}
}
