package com.aote.webmeter.tools.iot;

import com.af.plugins.HttpConnectionPoolUtil;
import com.aote.redis.RedisUtil;
import com.aote.webmeter.enums.IOTBusinessTypeEnum;
import com.aote.webmeter.enums.WebmeterPropertiesIOTEnum;
import com.aote.webmeter.tools.WebMeterInfo;
import com.iotplatform.client.NorthApiException;
import com.iotplatform.constant.AuthConstant;
import com.iotplatform.constant.ExceptionEnum;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.log4j.Logger;
import org.json.JSONObject;

import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;
import java.io.IOException;
import java.io.InputStream;
import java.security.KeyStore;

/**
 * 电信IOT平台 应用接口鉴权
 * @author Mr.river
 */
public class AuthenticationTools {

	private static final Logger LOGGER = Logger.getLogger(AuthenticationTools.class);

	/**
	 * 发送请求的客户端单例
 	 */
	private static volatile CloseableHttpClient httpClient;

	/**
	 * 获取HTTP客户端单例
	 */
	public static CloseableHttpClient getHttpClient(){
		if (httpClient == null){
			//多线程下同时调用getHttpClient容易导致重复创建httpClient对象的问题,所以加上了同步锁
			synchronized (AuthenticationTools.class){
				if (httpClient == null){
					try {
						httpClient = HttpConnectionPoolUtil.getHttpClient(initSslConfig());
					} catch (NorthApiException e) {
						e.printStackTrace();
					}
				}
			}
		}
		return httpClient;
	}

	/**
	 * 获取TOKEN
	 * @return TOKEN值
	 */
	public static String getAuthToken() {
		int isAep = IotCommonTools.getIsAep();
		String appId;
		String secret;
		if(isAep == 1){
			appId = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.AEP_APIKEY);
			secret = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.AEP_SECRET);
		} else if(isAep == 0){
			appId = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.OC_APIKEY);
			secret = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.OC_SECRET);
		} else {
			appId = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.OC2_APIKEY);
			secret = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.OC2_SECRET);
		}
		String token = getToken(appId);
		if (token != null) {
			return token;
		} else {
			return getWebAuthToken(appId, secret);
		}
	}

	/**
	 * 发起获取TOKEN的请求
	 *
	 * @return TOKEN信息
	 * @throws IOException IO异常
	 */
	public static synchronized String getWebAuthToken(String appId, String secret) {
		RedisUtil redisUtil = RedisUtil.getInstance();
		final String[] newAccessToken = new String[1];
		redisUtil.lock(appId,()->{
			String token = getToken(appId);
			if (token == null) {
				try {
					newAccessToken[0] = requestGetToken(appId, secret);
				} catch (IOException e) {
					e.printStackTrace();
				}
			} else {
				newAccessToken[0] = token;
			}
		});
		return newAccessToken[0];
	}

	private static String requestGetToken(String appId, String secret) throws IOException {
		RedisUtil redisUtil = RedisUtil.getInstance();
		JSONObject result;
		//组织AppId和密钥
		JSONObject content = new JSONObject();
		content.put("appId", appId);
		content.put("secret", secret);
		//先尝试获取刷新鉴权密钥
		String refreshToken = getRefreshToken(appId);
		if(refreshToken != null){
			LOGGER.info(appId + ":进行刷新鉴权操作");
			content.put("refreshToken", refreshToken);
			result = IotCommonTools.post(IOTBusinessTypeEnum.REFRESH_AUTH, new HttpPost(), content, null);
			if(result.has("errorEntity")){
				//如果刷新鉴权失败，则清理刷新鉴权密钥，重新鉴权
				redisUtil.delete(appId+"_refreshToken");
				return requestGetToken(appId, secret);
			} else {
				//取到新密钥和刷新密钥
				String newAccessToken = result.getString("accessToken");
				String newRefreshToken = result.getString("refreshToken");
				redisUtil.set(appId+"_token", newAccessToken, result.getInt("expiresIn") - 60);
				//如果取到的刷新密钥和之前的不一致，直接进行更新
				if(!refreshToken.equals(newRefreshToken)){
					redisUtil.set(appId+"_refreshToken", newRefreshToken, (3600 * 24) - 60);
				}
				return newAccessToken;
			}
		} else {
			LOGGER.info(appId + ":进行鉴权操作");
			result = IotCommonTools.post(IOTBusinessTypeEnum.AUTH, new HttpPost(), content, null);
			if(result.has("errorEntity")){
				//如果鉴权都失败了，那只能抛出异常了
				throw new RuntimeException(result.toString());
			}
			String newRefreshToken = result.getString("refreshToken");
			//存储刷新鉴权用的密钥，然后再次调用一遍该方法进行刷新鉴权，避免取到旧的密钥
			redisUtil.set(appId+"_refreshToken", newRefreshToken, (3600 * 24) - 60);
			return requestGetToken(appId, secret);
		}
	}

	private static String getToken(String appId) {
		RedisUtil redisUtil = RedisUtil.getInstance();
		Object token = redisUtil.get(appId+"_token");
		if(token != null){
			LOGGER.info(appId + ":获取现有token");
			return String.valueOf(token);
		} else {
			return null;
		}
	}

	private static String getRefreshToken(String appId) {
		RedisUtil redisUtil = RedisUtil.getInstance();
		Object token = redisUtil.get(appId+"_refreshToken");
		if(token != null){
			return String.valueOf(token);
		} else {
			return null;
		}
	}

	/**
	 * 初始化SSL证书
	 *
	 * @return SSL连接工厂
	 * @throws NorthApiException IOT平台请求异常
	 */
	protected static SSLConnectionSocketFactory initSslConfig() throws NorthApiException {
		try (InputStream isTrustCa = SignalDeliveryTools.class.getResourceAsStream("/ca.jks");
			 InputStream isSelfCert = SignalDeliveryTools.class.getResourceAsStream("/outgoing.CertwithKey.pkcs12")) {
			KeyStore selfCert = KeyStore.getInstance("pkcs12");
			selfCert.load(isSelfCert, AuthConstant.SELFCERTPWD.toCharArray());
			KeyManagerFactory kmf = KeyManagerFactory.getInstance("sunx509");
			kmf.init(selfCert, AuthConstant.SELFCERTPWD.toCharArray());
			KeyStore caCert = KeyStore.getInstance("jks");
			caCert.load(isTrustCa, AuthConstant.TRUSTCAPWD.toCharArray());
			TrustManagerFactory tmf = TrustManagerFactory.getInstance("sunx509");
			tmf.init(caCert);
			SSLContext sc = SSLContext.getInstance("TLS");
			sc.init(kmf.getKeyManagers(), tmf.getTrustManagers(), null);
			return new SSLConnectionSocketFactory(sc, NoopHostnameVerifier.INSTANCE);
		} catch (Exception var10) {
			throw new NorthApiException(ExceptionEnum.CLIENT_SSL_CONFIG_ERROR, var10.getMessage());
		}
	}
}
