package com.aote.webmeter.tools;

import com.aote.module.ModuleMapper;
import com.aote.webmeter.enums.*;
import com.aote.webmeter.exception.WebmeterException;
import com.aote.webmeter.tools.sql.DataSourceDialect;
import com.aote.webmeter.tools.sql.MySQLDialect;
import com.aote.webmeter.tools.sql.OracleDialect;
import com.aote.webmeter.tools.sql.SqlServerDialect;
import org.apache.log4j.Logger;

import java.util.HashMap;
import java.util.Map;

/**
 * 物联网表模块参数工具类
 * moduleName用于获取参数时指定模块
 * @author Mr.river
 */
public class WebMeterInfo {
    private static final Logger LOGGER = Logger.getLogger(WebMeterInfo.class);
    /**
     * 物联网表模块参数文件
     */
    private static final Map<String, HashMap<String,String>> PROPERTIES_MAP;
    /**
     * 物联网表模块名
     */
    private static final ThreadLocal<String> MODULE_NAME;
    /**
     * 物联网表营收版本名
     */
    private static SaleVersionEnum SALE_NAME;
    /**
     * 物联网表数据源类型
     */
    private static DataSourceTypeEnum DATA_SOURCE_TYPE;
    /**
     * 物联网表-数据源方言兼容性组件
     */
    public static final DataSourceDialect DATA_SOURCE_DIALECT;

    static {
        MODULE_NAME = new ThreadLocal<>();
        PROPERTIES_MAP = new HashMap<>(1);
        // 获取物联网表配置文件并加载
        load();
        // 初始化物联网表全局参数
        initWebmeterParams();
        // 初始化数据源方言兼容性组件
        if(DATA_SOURCE_TYPE == DataSourceTypeEnum.SQLSERVER){
            DATA_SOURCE_DIALECT = new SqlServerDialect();
        } else if(DATA_SOURCE_TYPE == DataSourceTypeEnum.ORACLE){
            DATA_SOURCE_DIALECT = new OracleDialect();
        } else {
            DATA_SOURCE_DIALECT = new MySQLDialect();
        }
    }

    /**
     * 获取物联网表配置文件并加载
     */
    public static void load() {
        // 获取所有引入的模块
        Map<String, Map<String, String>> map = ModuleMapper.getMap();
        for (String moduleName : map.keySet()) {
            //获取物联网表模块配置文件
            HashMap<String, String> config;
            String modulePath = map.get(moduleName).get("path");
            if (modulePath != null) {
                config = WebmeterPropertiesTools.readWebMeter(modulePath, moduleName);
            } else {
                config = WebmeterPropertiesTools.readWebMeter(moduleName);
            }
            if (config != null) {
                PROPERTIES_MAP.put(moduleName, config);
                LOGGER.debug("【" + moduleName + "】模块参数文件已加载");
            }
        }
    }

    /**
     * 清理物联网表配置文件
     */
    public static void clear() {
        PROPERTIES_MAP.clear();
    }

    /**
     * 设置物联网表模块
     *
     * @param name 物联网表模块名称
     */
    public static void setModuleName(String name) {
        if (PROPERTIES_MAP.containsKey(name)) {
            MODULE_NAME.set(name);
        } else {
            throw new WebmeterException(WebmeterExceptionCodeEnum.MODULE_NOT_FIND_FAIL,"模块："+name);
        }
    }

    /**
     * 初始化物联网表全局参数
     */
    public static void initWebmeterParams() throws WebmeterException{
        HashMap<String, String> config = WebmeterPropertiesTools.readWebMeter(ModuleEnum.PUBLIC_MODULE.getValue());
        // 设置营收版本值
        String saleNameParam = WebmeterPropertiesEnum.SALE_NAME.getValue();
        if (config.containsKey(saleNameParam)) {
            String saleName = config.get(saleNameParam);
            if (!SaleVersionEnum.is(saleName)) {
                throw new WebmeterException(WebmeterExceptionCodeEnum.SALE_VERSION_NOT_FIND_FAIL);
            }
            SALE_NAME = SaleVersionEnum.toType(saleName);
        } else {
            throw new WebmeterException(WebmeterExceptionCodeEnum.SALE_VERSION_NULL_FAIL);
        }
        // 设置数据源类型
        String dataSourceParam = WebmeterPropertiesEnum.DATA_SOURCE_TYPE.getValue();
        if (config.containsKey(dataSourceParam)) {
            String dataSourceType = config.get(dataSourceParam);
            if (!DataSourceTypeEnum.is(dataSourceType)) {
                throw new WebmeterException(WebmeterExceptionCodeEnum.DATA_SOURCE_TYPE_NOT_FIND_FAIL);
            }
            DATA_SOURCE_TYPE = DataSourceTypeEnum.toType(dataSourceType);
        } else {
            throw new WebmeterException(WebmeterExceptionCodeEnum.DATA_SOURCE_TYPE_NULL_FAIL);
        }
    }

    /**
     * 获取值
     * @param key key
     * @return value
     */
    public static String getString(String key){
        return PROPERTIES_MAP.get(MODULE_NAME.get()).get(key);
    }

    /**
     * 根据属性枚举获取值
     * @param keyValueEnum 属性枚举
     * @return value
     */
    public static String getString(PropertiesKeyValueEnum<String> keyValueEnum){
        return PROPERTIES_MAP.get(MODULE_NAME.get()).get(keyValueEnum.getValue());
    }

    /**
     * 获取值（取不到时取默认值）
     * @param key key
     * @param defaultValue 默认值
     * @return value
     */
    public static String getString(String key, String defaultValue){
        String value = PROPERTIES_MAP.get(MODULE_NAME.get()).get(key);
        return value != null ? value : defaultValue;
    }

    /**
     * 获取当前模块的参数Map
     * @return value
     */
    public static Map<String, String> getMap(){
        return PROPERTIES_MAP.get(MODULE_NAME.get());
    }

    /**
     * 获取指定模块的值
     * @param key key
     * @return value
     */
    public static String getStringByModule(String moduleName, String key){
        return PROPERTIES_MAP.get(moduleName).get(key);
    }

    /**
     * 获取当前模块名
     * @return moduleName
     */
    public static String getModuleName() {
        return MODULE_NAME.get();
    }

    /**
     * 获取当前营收版本名称
     * @return saleName
     */
    public static String getSaleName() {
        return SALE_NAME.getValue();
    }

    /**
     * 获取当前数据源类型
     * @return dataSourceType
     */
    public static String getDataSourceType() {
        return DATA_SOURCE_TYPE.getValue();
    }

    /**
     * 获取当前营收版本名称枚举
     * @return saleNameEnum
     */
    public static SaleVersionEnum getSaleNameEnum() throws WebmeterException{
        SaleVersionEnum saleVersionEnum = SALE_NAME;
        if(saleVersionEnum == null){
            throw new WebmeterException(WebmeterExceptionCodeEnum.SALE_VERSION_NULL_FAIL,"模块："+MODULE_NAME.get());
        }
        return saleVersionEnum;
    }

    /**
     * 获取当前数据源类型枚举
     * @return dataSourceTypeEnum
     */
    public static DataSourceTypeEnum getDataSourceTypeEnum() throws WebmeterException{
        DataSourceTypeEnum dataSourceTypeEnum = DATA_SOURCE_TYPE;
        if(dataSourceTypeEnum == null){
            throw new WebmeterException(WebmeterExceptionCodeEnum.DATA_SOURCE_TYPE_NULL_FAIL,"模块："+MODULE_NAME.get());
        }
        return dataSourceTypeEnum;
    }

    /**
     * 判断是否含有该key
     * @param key key
     * @return isHas
     */
    public static Boolean has(String key){
        return PROPERTIES_MAP.get(MODULE_NAME.get()).containsKey(key);
    }

    /**
     * 根据属性枚举判断是否含有该key
     * @param keyValueEnum 属性枚举
     * @return value
     */
    public static Boolean has(PropertiesKeyValueEnum<String> keyValueEnum){
        return PROPERTIES_MAP.get(MODULE_NAME.get()).containsKey(keyValueEnum.getValue());
    }

    /**
     * 判断指定模块是否含有该key
     * @param key key
     * @return isHas
     */
    public static Boolean has(String module, String key){
        return PROPERTIES_MAP.get(module).containsKey(key);
    }

    /**
     * 根据属性枚举判断指定模块是否含有该key
     * @param keyValueEnum 属性枚举
     * @return value
     */
    public static Boolean has(String module, PropertiesKeyValueEnum<String> keyValueEnum){
        return PROPERTIES_MAP.get(module).containsKey(keyValueEnum.getValue());
    }

    /**
     * 清除设置的物联网表模块
     */
    public static void removeModuleName(){
        MODULE_NAME.remove();
    }

    /**
     * 判断是否含有该模块
     * @param key key
     * @return isHas
     */
    public static Boolean hasModule(String key){
        return PROPERTIES_MAP.containsKey(key);
    }

    /**
     * 该方法为内部方法，仅用于foreignaid模块
     */
    public static void setSaleName(String saleName){
        SALE_NAME = SaleVersionEnum.toType(saleName);
    }
}
