package com.aote.webmeter.common.utils;

/**
 * SQL条件表达式生成器
 *
 * @author Mr.river
 */
public class Condition{
    /**
     * SQL语法标识
     * 当你的参数值是SQL语法而非具体值时，在你的参数值加上此前缀，以方便区分
     */
    public static final String SQL_GRAMMAR_TAG = "@";

    /**
     * 生成结果
     */
    private final StringBuilder value;

    public static Condition build(){
        return new Condition();
    }

    private Condition(){
        value = new StringBuilder();
    }

    private Condition(String initValue){
        value = new StringBuilder(initValue);
    }

    public String getValue(){
        return value.toString();
    }

    /**
     * 简单拼接
     * @param column 数据列
     * @param conditionType 类型
     * @param value 值
     * @return 条件表达式生成器
     */
    private Condition simpleAppend(String column,String conditionType, Object value){
        String valueStr = String.valueOf(value);
        if(valueStr.startsWith(SQL_GRAMMAR_TAG)){
            valueStr = valueStr.substring(1);
            this.value.append(column).append(" ").append(conditionType).append(" ").append(valueStr);
        } else {
            this.value.append(column).append(" ").append(conditionType).append(" ")
                    .append("'").append(valueStr).append("'");
        }
        return this;
    }

    /**
     * IN类型拼接
     * @param column 数据列
     * @param conditionType 类型
     * @param value 一些值
     * @return 条件表达式生成器
     */
    private Condition inAppend(String column, String conditionType, Object... value){
        this.value.append(column).append(" ").append(conditionType).append(" (");
        for(Object item : value){
            this.value.append("'").append(item).append("'").append(",");
        }
        this.value.deleteCharAt(this.value.length() - 1).append(")");
        return this;
    }

    public Condition eq(String column, Object value){
        return simpleAppend(column, "=", value);
    }

    public Condition neq(String column, Object value){
        return simpleAppend(column, "!=", value);
    }

    public Condition greaterThan(String column, Object value){
        return simpleAppend(column, ">", value);
    }

    public Condition greaterThanEq(String column, Object value){
        return simpleAppend(column, ">=", value);
    }

    public Condition lessThan(String column, Object value){
        return simpleAppend(column, "<", value);
    }

    public Condition lessThanEq(String column, Object value){
        return simpleAppend(column, "<=", value);
    }

    public Condition like(String column, Object value){
        return simpleAppend(column, "LIKE", "%" + value + "%");
    }

    public Condition leftLike(String column, Object value){
        return simpleAppend(column, "LIKE", "%" + value);
    }

    public Condition rightLike(String column, Object value){
        return simpleAppend(column, "LIKE", value + "%");
    }

    public Condition in(String column, Object... value){
        return inAppend(column,"IN",value);
    }

    public Condition notIn(String column, Object... value){
        return inAppend(column,"NOT IN",value);
    }

    public Condition isNull(String column){
        this.value.append(column).append(" ").append("IS NULL");
        return this;
    }

    public Condition isNotNull(String column){
        this.value.append(column).append(" ").append("IS NOT NULL");
        return this;
    }

    public Condition and(){
        this.value.append("\n\t  AND ");
        return this;
    }

    public Condition or(){
        this.value.append("\n\t  OR ");
        return this;
    }

    /**
     * 置为优先判断
     * @param condition 条件表达式生成器
     * @return 条件表达式生成器
     */
    public Condition asFirst(Condition condition){
        this.value.append("(").append(condition.value).append(")");
        return this;
    }

    @Override
    public String toString(){
        return this.value.toString();
    }
}
