package com.aote.webmeter.common.template.saveInstruct;

import com.aote.util.JsonHelper;
import com.aote.webmeter.common.basic.manage.InstructManage;
import com.aote.webmeter.common.dao.ChangeMeterDataDao;
import com.aote.webmeter.common.dao.SaleDataDao;
import com.aote.webmeter.common.dao.UserFilesDao;
import com.aote.webmeter.common.dto.GetRechargeDataDto;
import com.aote.webmeter.common.stereotype.Template;
import com.aote.webmeter.common.template.AbstractTemplateBuilder;
import com.aote.webmeter.common.template.pour.SaveTemplatePour;
import com.aote.webmeter.enums.BusinessExceptionCodeEnum;
import com.aote.webmeter.enums.business.InstructInputtorEnum;
import com.aote.webmeter.enums.business.InstructTypeEnum;
import com.aote.webmeter.enums.business.SaleTypeEnum;
import com.aote.webmeter.exception.WebmeterException;
import com.aote.webmeter.tools.SpringBeanUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Optional;

/**
 * 保存充值指令模板
 * @author Mr.river
 */
@Template
public class SaveGasRechargeInstructTemplate extends BasicSaveInstructTemplate {
    private static final Logger LOGGER = LoggerFactory.getLogger(SaveGasRechargeInstructTemplate.class);

    /**
     * 查询待生成指令的充值数据-Result
     */
    private GetRechargeDataDto dto;
    /**
     * 调用保存指令模板时注入业务
     */
    private SaveTemplatePour saveTemplatePour;
    /**
     * 是否用于余额同步的兼容，默认false，
     * 用于某些系统计费表采用非实时结算，要求在充值时下发余额同步的情况，例如积成的一些系统计费表，
     * 个人认为相比于实时结算，这是一种冗余过时的后台计费方式，仅用于兼容
     */
    private Boolean isSyncBalance;


    protected SaveGasRechargeInstructTemplate(InstructManage instructManage, UserFilesDao userFilesDao, SaleDataDao saleDataDao, ChangeMeterDataDao changeMeterDataDao){
        super(instructManage,userFilesDao,saleDataDao, changeMeterDataDao);
    }


    public static class Builder extends AbstractTemplateBuilder {
        private final SaveGasRechargeInstructTemplate template;

        public Builder(GetRechargeDataDto dto, SaveTemplatePour saveTemplatePour) {
            template = SpringBeanUtil.getBean(SaveGasRechargeInstructTemplate.class);
            template.dto = dto;
            template.saveTemplatePour = saveTemplatePour;
        }

        @Override
        public SaveGasRechargeInstructTemplate build() {
            template.isSyncBalance = Optional.ofNullable(template.isSyncBalance).orElse(false);
            return template;
        }

        public Builder moduleName(String moduleName){
            template.moduleName = moduleName;
            return this;
        }

        public Builder isSyncBalance(Boolean isSyncBalance){
            template.isSyncBalance = isSyncBalance;
            return this;
        }
    }

    @Override
    protected void beforeExec() {
        super.userId = dto.getUserId();
        if(isSyncBalance){
            super.type = InstructTypeEnum.ChargeMeter;
        } else {
            super.type = InstructTypeEnum.GasRecharge;
        }
        super.content = JsonHelper.toJSON(saveTemplatePour.run(dto));
        super.inputtor = InstructInputtorEnum.CHARGE;
        super.title = dto.getType().getValue();
        super.dataId = dto.getId();
    }

    @Override
    protected void afterExec() {
        //更新充值记录
        SaleTypeEnum saleTypeEnum = dto.getType();
        switch (saleTypeEnum){
            case CHARGE:
            case REFUND:
            case CHARGE_OR_REFUND:
                saleDataDao.updateSellingGasSyncState(dto.getId());
                break;
            case CANCEL_CHARGE:
                saleDataDao.updateCancelSellingGasSyncState(dto.getId());
                break;
            case ADD_FEE:
            case SUB_FEE:
            case ADD_OR_SUB_FEE:
                saleDataDao.updateAdjustMoneySyncState(dto.getId());
                break;
            case REPLACE_METER_ADD_VOLUME:
                saleDataDao.updateSupplyGasSyncState(dto.getId());
                break;
            default:
                throw new WebmeterException(BusinessExceptionCodeEnum.INVALID_RECHARGE_SOURCE_TYPE);

        }
    }
}
