package com.aote.webmeter.common.dao;

import com.aote.sql.SqlServer;
import com.aote.webmeter.common.utils.Condition;
import com.aote.webmeter.enums.SaleVersionEnum;
import com.aote.webmeter.enums.business.OpenAccountStateEnum;
import com.aote.webmeter.exception.DataExceptionHandle;
import com.aote.webmeter.exception.DefaultUserFilesDataHandle;
import com.aote.webmeter.tools.CompatTools;
import com.aote.webmeter.tools.WebMeterInfo;
import lombok.SneakyThrows;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.HashMap;
import java.util.Optional;

/**
 * 表档案数据库操作
 *
 * @author Mr.river
 */
@Transactional(rollbackFor=Exception.class)
@Component
public class UserFilesDao extends AbstractDao {
    private final SqlServer sqlServer;

    public UserFilesDao(SqlServer sqlServer) {
        this.sqlServer = sqlServer;
    }

    /**
     * 创建指令时获取档案信息
     * @param userId 表档案ID
     * @param alias 表品牌别名
     * @param handle 数据异常处理
     * @return 档案信息
     * @throws Exception 查询异常
     */
    @SneakyThrows
    public JSONArray getMeterInfoByCreateInstruct(String userId, String alias, DataExceptionHandle handle) {
        //获取条件表达式
        String condition = CompatTools.getBasicCondition(userId);
        //获取营收系统版本名
        SaleVersionEnum saleName = WebMeterInfo.getSaleNameEnum();
        //获取SQL名称
        String sqlName;
        switch (saleName){
            case SILVER_LIGHT:
                sqlName = "yinGuangGetMeterInfoBySaveInstruct";
                break;
            case SILVER_LIGHT2:
                sqlName = "yinGuangGetMeterInfo2BySaveInstruct";
                break;
            default:
                sqlName = "getMeterInfoBySaveInstruct";
                break;
        }
        //查询用户档案，获取生成指令所需的信息
        JSONObject sqlParams = new JSONObject();
        sqlParams.put("f_meter_brand", alias);
        sqlParams.put("condition", condition);
        JSONArray result = sqlServer.query(sqlName,sqlParams);
        //检查数据
        Optional.ofNullable(handle).orElseGet(DefaultUserFilesDataHandle::new).checkData(result, condition);
        return result;
    }

    /**
     * 更新表具开户状态
     * @param userId 表档案ID
     */
    public void updateMeterOpenAccountState(String userId, OpenAccountStateEnum stateEnum){
        sqlServer.run(CompatTools.getUpdateUserFilesSql(userId, stateEnum));
    }

    /**
     * 更新NB表具平台设备号
     * @param meterNumber 表号
     * @param deviceId 设备号
     */
    public void updateMeterDeviceId(String meterNumber, String deviceId){
        String tableName = "t_meterinfo";
        HashMap<String,Object> params = new HashMap<>(1);
        params.put("f_meternumber", meterNumber);
        sqlServer.run(formatUpdateSql(tableName,params,
                Condition.build().eq("f_device_id", deviceId),
                true));
    }
}
