package com.aote.webmeter.tools.iot.onenet;

import com.aote.webmeter.tools.WebMeterInfo;
import com.aote.webmeter.tools.iot.utils.AuthTools;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.Base64;

/**
 * 移动Onenet-IOT平台 API接口鉴权
 *
 * @author Mr.river
 */
@Component
public class OnenetApiAuthTools extends AuthTools<JSONObject> {

	private static final Logger LOGGER = LoggerFactory.getLogger(OnenetApiAuthTools.class);

	/**
	 * 超时时间值 60分钟
	 */
	private static final Integer TOKEN_TIMEOUT_VALUE = 3600;

	@Override
	protected AuthResult getToken(String orgId) throws UnsupportedEncodingException {
		String version = "2022-05-01";
		String resourceName = "products/" + WebMeterInfo.getString("onenetProductId");
		String expirationTime = System.currentTimeMillis() / 1000 + 100 * 24 * 60 * 60 + "";
		String signatureMethod = SignatureMethod.SHA1.name().toLowerCase();
		String accessKey = WebMeterInfo.getString("onenetProductAccessKey");
		StringBuilder sb = new StringBuilder();
		String res = URLEncoder.encode(resourceName, String.valueOf(StandardCharsets.UTF_8));
		// generatorSignature
		String encryptText = expirationTime + "\n" + signatureMethod + "\n" + resourceName + "\n" + version;
		String signature;
        byte[] bytes = null;
        try {
            bytes = HmacEncrypt(encryptText, accessKey, signatureMethod);
        } catch (NoSuchAlgorithmException | InvalidKeyException e) {
            throw new RuntimeException(e);
        }
        signature = Base64.getEncoder().encodeToString(bytes);
		String sig = URLEncoder.encode(signature, String.valueOf(StandardCharsets.UTF_8));
		// append header
		sb.append("version=")
				.append(version)
				.append("&res=")
				.append(res)
				.append("&et=")
				.append(expirationTime)
				.append("&method=")
				.append(signatureMethod)
				.append("&sign=")
				.append(sig);
		JSONObject header = new JSONObject();
		header.put("Authorization", sb.toString());
        return new AuthResult(header, TOKEN_TIMEOUT_VALUE - 60);
	}

	private static byte[] HmacEncrypt(String data, String key, String signatureMethod)
			throws NoSuchAlgorithmException, InvalidKeyException {
		//根据给定的字节数组构造一个密钥,第二参数指定一个密钥算法的名称
		SecretKeySpec signinKey;
		signinKey = new SecretKeySpec(Base64.getDecoder().decode(key),
				"Hmac" + signatureMethod.toUpperCase());
		//生成一个指定 Mac 算法 的 Mac 对象
		Mac mac = Mac.getInstance("Hmac" + signatureMethod.toUpperCase());
		//用给定密钥初始化 Mac 对象
		mac.init(signinKey);
		//完成 Mac 操作
		return mac.doFinal(data.getBytes());
	}

	@Override
	protected String getName() {
		return "Onenet";
	}

	public enum SignatureMethod {
		SHA1, MD5, SHA256;
	}
}
