package com.aote.webmeter.tools.iot.onenet;

import com.af.plugins.HttpAsyncConnectionPoolUtil;
import com.af.plugins.HttpConnectionPoolUtil;
import com.af.plugins.RestTools;
import com.aote.webmeter.enums.OnenetApiEnum;
import com.aote.webmeter.enums.WebmeterPropertiesIOTEnum;
import com.aote.webmeter.tools.WebMeterInfo;
import lombok.SneakyThrows;
import org.apache.http.client.methods.HttpPost;
import org.apache.log4j.Logger;
import org.json.JSONObject;

import java.io.IOException;

/**
 * 移动Onenet-IOT平台 API调用工具
 * @author Mr.river
 */
public class OnenetRestTools {
    private static final Logger LOGGER = Logger.getLogger(OnenetRestTools.class);

    /**
     * 创建设备
     * @param params 请求参数
     * @param imei imei码
     * @param imsi imsi码
     * @return 响应结果
     */
    @SneakyThrows
    public static JSONObject createDevice(JSONObject params, String imei, String imsi) {
        //指定请求路径
        String path = OnenetApiEnum.DEVICE_URL.getValue();
        //设置请求头
        JSONObject headers = new JSONObject();
        headers.put("api-key", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_APIKEY));
        //设置imei和imsi
        JSONObject authInfo = new JSONObject();
        authInfo.put(imei, imsi);
        params.put("auth_info", authInfo);
        //发送请求
        JSONObject result = new JSONObject(HttpConnectionPoolUtil.request(path,
                params.toString(),
                headers.toString(),
                new HttpPost()
        ));
        int errno = result.getInt("errno");
        String msg;
        if(errno == 0){
            msg = result.getJSONObject("data").getString("device_id");
        } else {
            if(errno == 6){
                msg = "尝试注册时已绑定";
            } else {
                msg = "注册IOT设备失败："+result.getString("error");
            }
            errno = -1;
        }
        return new JSONObject().put("code",errno).put("msg",msg);
    }

    /**
     * 移动物联网设备
     * @param params
     * @return
     */
    @SneakyThrows
    public static JSONObject createMobileDevice(JSONObject params, String imei, String imsi) {
        //指定请求路径
        String path = OnenetApiEnum.MOBILE_DEVICE_URL.getValue()+"/create";
        params.put("product_id",WebMeterInfo.getString("onenetAppId"));
        params.put("device_name",params.getString("title"));
        params.put("imei",imei);
        params.put("imsi",imsi);
        //设置请求头
        JSONObject headers = new JSONObject();
        headers.put("Content-Type", "application/json");
        headers.put("Authorization", OnenetAuthenticationTools.getAuthToken());
        //发送请求
        JSONObject result = new JSONObject(HttpConnectionPoolUtil.request(path,
                params.toString(),
                headers.toString(),
                new HttpPost()
        ));
        int errno = result.getInt("code");
        String msg;
        if(errno == 0){
            msg = result.getJSONObject("data").getString("sec_key");
        } else {
            if(errno == 10406){
                msg = "新增设备失败：设备已存在";
            } else {
                msg = "注册IOT设备失败："+result.getString("msg");
            }
            errno = -1;
        }
        return new JSONObject().put("code",errno).put("msg",msg);
    }

    /**
     * 获取设备信息
     *
     * @param deviceId 设备ID
     * @return 响应结果
     */
    public static JSONObject getDevice(String deviceId) {
        //指定请求路径
        String path = OnenetApiEnum.DEVICE_URL.getValue() + "/" + deviceId;
        //设置请求头
        JSONObject headers = new JSONObject();
        headers.put("api-key", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_APIKEY));
        //发送请求
        return new JSONObject(RestTools.get(path, headers));
    }

    /**
     * 删除设备
     *
     * @param deviceId 设备ID
     * @return 响应结果
     */
    public static JSONObject deleteDevice(String deviceId) {
        //指定请求路径
        String path = OnenetApiEnum.DEVICE_URL.getValue() + "/" + deviceId;
        //设置请求头
        JSONObject headers = new JSONObject();
        headers.put("api-key", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_APIKEY));
        //发送请求
        JSONObject result = new JSONObject(RestTools.delete(path, null, headers));
        int errno = result.getInt("errno");
        String msg;
        if(errno == 0){
            msg = "成功";
        } else {
            if(errno == 3){
                errno = 0;
                msg = "尝试删除时已不存在";
            } else {
                errno = -1;
                msg = "删除IOT设备失败："+result.getString("error");
            }
        }
        return new JSONObject().put("code",errno).put("msg",msg);
    }

    /**
     * 删除移动设备
     * @param params
     * @return
     */
    public static JSONObject deleteMobileDevice(JSONObject params) throws IOException {
        //指定请求路径
        String path = OnenetApiEnum.MOBILE_DEVICE_URL.getValue() + "/delete";
        params.put("product_id",WebMeterInfo.getString("onenetAppId"));
        params.put("device_name",params.getString("meternumber"));
        //发送请求
        JSONObject headers = new JSONObject();
        headers.put("Content-Type", "application/json");
        headers.put("Authorization", OnenetAuthenticationTools.getAuthToken());
        //发送请求
        JSONObject result = new JSONObject(HttpConnectionPoolUtil.request(path,
                params.toString(),
                headers.toString(),
                new HttpPost()
        ));
        int errno = result.getInt("code");
        String msg;
        if(errno == 0){
            msg = "成功";
        } else {
            if(errno == 10410){
                errno = 0;
                msg = "尝试删除时已不存在";
            } else {
                errno = -1;
                msg = "删除IOT设备失败："+result.getString("msg");
            }
        }
        return new JSONObject().put("code",errno).put("msg",msg);
    }

    /**
     * 读取设备资源
     * @param imei 设备imei
     * @param timeout 超时时间，默认25秒
     * @return 响应结果
     */
    public static JSONObject readDeviceRes(String imei, String timeout) {
        JSONObject params = new JSONObject();
        params.put("imei", imei);
        params.put("obj_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_ID));
        params.put("obj_inst_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_INST_ID));
        params.put("res_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_RES_ID));
        params.put("timeout", timeout == null ? "25" : timeout);
        //指定请求路径
        String path = RestTools.formatURL(OnenetApiEnum.RESOURCE_URL.getValue(), params);
        //设置请求头
        JSONObject headers = new JSONObject();
        headers.put("api-key", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_APIKEY));
        //发送请求
        return new JSONObject(RestTools.get(path, headers));
    }

    /**
     * 写入设备资源
     * @param imei 设备imei
     * @param mode 写入模式（1或2）
     * @param timeout 超时时间，默认25秒
     * @param content 请求内容(实际请求参数)
     * @return 响应结果
     */
    @SneakyThrows
    public static JSONObject writeDeviceRes(String imei, String mode, String timeout, JSONObject content) {
        JSONObject params = new JSONObject();
        params.put("imei", imei);
        params.put("obj_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_ID));
        params.put("obj_inst_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_INST_ID));
        params.put("mode", mode == null ? 1 : mode);
        params.put("timeout", timeout == null ? "25" : timeout);
        //指定请求路径
        String path = RestTools.formatURL(OnenetApiEnum.RESOURCE_URL.getValue(), params);
        //设置请求头
        JSONObject headers = new JSONObject();
        headers.put("api-key", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_APIKEY));
        //发送请求
        LOGGER.debug("请求的地址："+ path);
        LOGGER.debug("请求头："+ headers);
        LOGGER.debug("请求体："+ content);
        HttpAsyncConnectionPoolUtil.request(path,
                content.toString(),
                headers.toString(),
                new HttpPost());
        return new JSONObject().put("errno",0);
    }

    /**
     * 命令下发
     * @param imei 设备imei
     * @param timeout 超时时间，默认25秒
     * @param content 请求内容(实际请求参数)
     * @return 响应结果
     */
    @SneakyThrows
    public static JSONObject sendCommand(String imei, String timeout, JSONObject content) {
        JSONObject params = new JSONObject();
        params.put("imei", imei);
        params.put("obj_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_ID));
        params.put("obj_inst_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_INST_ID));
        params.put("res_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_RES_ID));
        params.put("timeout", timeout == null ? "25" : timeout);
        //指定请求路径
        String path = RestTools.formatURL(OnenetApiEnum.COMMAND_URL.getValue(), params);
        //设置请求头
        JSONObject headers = new JSONObject();
        headers.put("api-key", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_APIKEY));
        //发送请求
        LOGGER.debug("请求的地址："+ path);
        LOGGER.debug("请求头："+ headers);
        LOGGER.debug("请求体："+ content);
        HttpAsyncConnectionPoolUtil.request(path,
                content.toString(),
                headers.toString(),
                new HttpPost());
        return new JSONObject().put("errno",0);
    }

    /**
     * 命令下发
     * @param imei 设备imei
     * @param timeout 超时时间，默认25秒
     * @param content 请求内容(实际请求参数)
     * @return 响应结果
     */
    @SneakyThrows
    public static JSONObject sendNewCommand(String imei, String timeout, JSONObject content) {
        JSONObject params = new JSONObject();
        params.put("imei", imei);
        params.put("obj_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_ID));
        params.put("obj_inst_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_INST_ID));
        params.put("res_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_RES_ID));
        params.put("timeout", timeout == null ? "25" : timeout);
        //指定请求路径
        String path = RestTools.formatURL(OnenetApiEnum.COMMAND_URL.getValue(), params);
        //设置请求头
        JSONObject headers = new JSONObject();
        headers.put("Authorization", OnenetAuthenticationTools.getAuthToken());
        //发送请求
        LOGGER.debug("请求的地址："+ path);
        LOGGER.debug("请求头："+ headers);
        LOGGER.debug("请求体："+ content);
        HttpAsyncConnectionPoolUtil.request(path,
                content.toString(),
                headers.toString(),
                new HttpPost());
        return new JSONObject().put("errno",0);
    }

    /**
     * 订阅
     *
     * @param params 请求参数
     * @return 响应结果
     */
    @SneakyThrows
    public static JSONObject subscribe(JSONObject params) {
        //指定请求路径
        String path = RestTools.formatURL(OnenetApiEnum.SUBSCRIPTIONS_URL.getValue(), params);
        //设置请求头
        JSONObject headers = new JSONObject();
        headers.put("api-key", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_APIKEY));
        //发送请求
        String result = HttpConnectionPoolUtil.request(path,
                null,
                headers.toString(),
                new HttpPost());
        return new JSONObject(result);
    }
}
