package com.aote.webmeter.tools.iot.onenet;

import com.aote.redis.RedisUtil;
import com.aote.webmeter.tools.WebMeterInfo;
import com.aote.webmeter.tools.iot.AuthenticationTools;
import org.apache.log4j.Logger;
import org.json.JSONObject;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.net.URLEncoder;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.Base64;

/**
 * @ClassName: IntelliJ IDEA
 * @Author: 林
 * @Date: 2023/10/12 16:23
 * @Description:
 */
public class OnenetAuthenticationTools {

    private static final Logger LOGGER = Logger.getLogger(AuthenticationTools.class);

    /**
     * 获取鉴权信息
     *
     * @return 鉴权信息
     */
    private static JSONObject getAuthInfo() {
        String appId = WebMeterInfo.getString("onenetAppId");
        String secret = WebMeterInfo.getString("onenetAPIKey");
        JSONObject result = new JSONObject();
        result.put("appId", appId);
        result.put("secret", secret);
        return result;
    }

    /**
     * 获取TOKEN
     *
     * @param force 是否强制刷新
     * @return TOKEN值
     */
    public static String getAuthToken(boolean force) {
        JSONObject authInfo = getAuthInfo();
        String appId = authInfo.getString("appId");
        String secret = authInfo.getString("secret");
        String token = getToken(appId, force);
        if (token != null) {
            return token;
        } else {
            return getWebAuthToken(appId, secret, force);
        }
    }

    public static String getAuthToken() {
        return getAuthToken(false);
    }

    /**
     * 请求获取token信息
     *
     * @param appId  appId
     * @param secret secret
     * @return TOKEN
     */
    private static String requestGetToken(String appId, String secret) throws Exception {
        RedisUtil redisUtil = RedisUtil.getInstance();
        String result;
        //组织AppId和密钥
        //先尝试获取刷新鉴权密钥
        LOGGER.info(appId + ":进行鉴权操作");
        result = getToken();
        LOGGER.info("得到的鉴权结果:"+result);
        String newRefreshToken = result;
        //存储鉴权用的密钥
        redisUtil.set(appId + "_token", newRefreshToken, (3600 * 24) - 60);
        return newRefreshToken;
    }

    /**
     * 获取Token
     *
     * @param appId appId
     * @param force 是否强制刷新Token
     * @return Token
     */
    private static String getToken(String appId, boolean force) {
        if (force) {
            return null;
        }
        RedisUtil redisUtil = RedisUtil.getInstance();
        Object token = redisUtil.get(appId + "_token");
        if (token != null) {
            LOGGER.info(appId + ":获取现有token");
            return String.valueOf(token);
        } else {
            return null;
        }
    }

    /**
     * 发起获取TOKEN的请求
     *
     * @param force 是否强制刷新
     * @return TOKEN信息
     */
    public static synchronized String getWebAuthToken(String appId, String secret, boolean force) {
        RedisUtil redisUtil = RedisUtil.getInstance();
        final String[] newAccessToken = new String[1];
        redisUtil.lock(appId, () -> {
            String token = getToken(appId, force);
            if (force) {
                LOGGER.info(appId + ":进行强制刷新鉴权操作");
            }
            if (token == null) {
                try {
                    newAccessToken[0] = requestGetToken(appId, secret);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            } else {
                newAccessToken[0] = token;
            }
        });
        return newAccessToken[0];
    }


    public static String assembleToken(String version, String resourceName, String expirationTime, String signatureMethod, String accessKey)
            throws Exception {
        StringBuilder sb = new StringBuilder();
        String res = URLEncoder.encode(resourceName, "UTF-8");
        String sig = URLEncoder.encode(generatorSignature(version, resourceName, expirationTime, accessKey, signatureMethod), "UTF-8");
        sb.append("version=")
                .append(version)
                .append("&res=")
                .append(res)
                .append("&et=")
                .append(expirationTime)
                .append("&method=")
                .append(signatureMethod)
                .append("&sign=")
                .append(sig);
        return sb.toString();
    }

    public static String generatorSignature(String version, String resourceName, String expirationTime, String accessKey, String signatureMethod)
            throws NoSuchAlgorithmException, InvalidKeyException {
        String encryptText = expirationTime + "\n" + signatureMethod + "\n" + resourceName + "\n" + version;
        String signature;
        byte[] bytes = HmacEncrypt(encryptText, accessKey, signatureMethod);
        signature = Base64.getEncoder().encodeToString(bytes);
        return signature;
    }

    public static byte[] HmacEncrypt(String data, String key, String signatureMethod)
            throws NoSuchAlgorithmException, InvalidKeyException {
        //根据给定的字节数组构造一个密钥,第二参数指定一个密钥算法的名称
        SecretKeySpec signinKey = null;
        signinKey = new SecretKeySpec(Base64.getDecoder().decode(key),
                "Hmac" + signatureMethod.toUpperCase());

        //生成一个指定 Mac 算法 的 Mac 对象
        Mac mac = null;
        mac = Mac.getInstance("Hmac" + signatureMethod.toUpperCase());

        //用给定密钥初始化 Mac 对象
        mac.init(signinKey);

        //完成 Mac 操作
        return mac.doFinal(data.getBytes());
    }

    public enum SignatureMethod {
        SHA1, MD5, SHA256;
    }

    public static String getToken() throws Exception {
        String version = "2022-05-01";
        String resourceName = "products/"+ WebMeterInfo.getString("onenetAppId");
        String expirationTime = System.currentTimeMillis() / 1000 + 24 * 60 * 60 + "";
        String signatureMethod = SignatureMethod.SHA1.name().toLowerCase();
        String accessKey = WebMeterInfo.getString("onenetAPIKey");
        return assembleToken(version,resourceName,expirationTime,signatureMethod,accessKey);
    }

    public static void main(String[] args) throws Exception {
        String version = "2022-05-01";
        String resourceName = "products/9uB0W1UFV1";
        String expirationTime = System.currentTimeMillis() / 1000 + 24 * 60 * 60 + "";
        String signatureMethod = SignatureMethod.SHA1.name().toLowerCase();
        String accessKey = "cwHn+9vNE5vvB9Ez9+NVUZ1/O6/BO+HFa8PFmeOd2yc=";
        System.out.println( assembleToken(version,resourceName,expirationTime,signatureMethod,accessKey));
    }
}
