package com.aote.webmeter.common.template.builder;

import com.aote.webmeter.common.basic.pour.PourSaveLogic;
import com.aote.webmeter.enums.InstructInputtorEnum;
import com.aote.webmeter.enums.InstructTypeEnum;
import lombok.Getter;
import org.json.JSONObject;

/**
 * 保存指令模板-抽象构造器
 * @author Mr.river
 */
@Getter
public abstract class SaveInstructTemplateBuilder extends TemplateBuilder {

    private String userId;
    private InstructTypeEnum type;
    private String content;
    private String dataId;
    private String title;
    private String alias;
    private InstructInputtorEnum inputtor = InstructInputtorEnum.SYSTEM_DEFAULT;
    private String moduleName;
    private JSONObject entity;
    private PourSaveLogic pourSaveLogic;

    protected SaveInstructTemplateBuilder(String userId, InstructTypeEnum type) {
        super();
        this.setUserId(userId);
        this.setType(type);
    }

    protected SaveInstructTemplateBuilder(JSONObject entity, InstructTypeEnum type, PourSaveLogic pourSaveLogic) {
        super();
        this.setUserId(String.valueOf(entity.get("userid")));
        this.setType(type);
        this.setEntity(entity);
        this.setPourSaveLogic(pourSaveLogic);
        this.setAlias(String.valueOf(entity.get("alias")));
        //获取指令内容
        String content = pourSaveLogic.run(entity).toString();
        this.setContent(content);
    }

    /**
     * @param userId 表档案主键值，即t_userfiles表的主键值
     * @apiNote 在不同营收系统版本下字段名不同，具体差异如下
     * @see com.aote.webmeter.tools.CompatTools#getOriginalUserFilesId()
     */
    protected SaveInstructTemplateBuilder setUserId(String userId) {
        this.userId = userId;
        return this;
    }

    /**
     * @param type 指令类型
     * @apiNote 指令类型枚举，用于区分不同指令
     * @see InstructTypeEnum
     */
    protected SaveInstructTemplateBuilder setType(InstructTypeEnum type) {
        this.type = type;
        return this;
    }

    /**
     * @param content 指令内容
     * @apiNote 组织的JSON格式指令内容
     */
    protected SaveInstructTemplateBuilder setContent(String content) {
        this.content = content;
        return this;
    }

    /**
     * @param dataId 操作记录ID
     * @return 指令的原操作记录编号
     */
    protected SaveInstructTemplateBuilder setDataId(String dataId) {
        this.dataId = dataId;
        return this;
    }

    /**
     * @param title 指令标题
     * @apiNote 对指令类型的补充说明
     */
    protected SaveInstructTemplateBuilder setTitle(String title) {
        this.title = title;
        return this;
    }

    /**
     * @param alias 表具品牌别名
     * @apiNote 指令归属的表具品牌别名
     */
    protected SaveInstructTemplateBuilder setAlias(String alias) {
        this.alias = alias;
        return this;
    }

    /**
     * @param inputtor 操作人
     * @apiNote 操作人枚举，亦用于需要用指令角色区分的业务场景
     * @see InstructInputtorEnum
     */
    protected SaveInstructTemplateBuilder setInputtor(InstructInputtorEnum inputtor) {
        this.inputtor = inputtor;
        return this;
    }

    /**
     * @param moduleName 物联网表模块名称
     * @apiNote 构建模板的物联网表模块名称
     * <p>在绝大多数业务场景中，该参数都是通过业务定时器注入，请参阅
     * {@link com.aote.webmeter.timer.WebMeterTimer#doBusiness(String, String, String, JSONObject)}
     * </p>
     */
    protected SaveInstructTemplateBuilder setModuleName(String moduleName) {
        this.moduleName = moduleName;
        return this;
    }

    /**
     * @param entity 生成指令所需的业务数据实体，即数据库查询出的记录
     * @apiNote 此处因为框架限制，所以使用宽泛类型JSONObject，后续应使用强类型增加程序健壮性
     * @see com.aote.webmeter.timer.WebMeterTimer#getSqlResult(JSONObject, String)
     */
    public SaveInstructTemplateBuilder setEntity(JSONObject entity) {
        this.entity = entity;
        return this;
    }

    /**
     * @param pourSaveLogic 保存指令所需的注入逻辑接口
     * @apiNote 该接口用于扩展实现组织指令内容的逻辑，run方法接收原始业务记录数据(entity)，
     * 返回一个JSONObject作为需要保存的指令内容
     * @see PourSaveLogic#run(JSONObject)
     */
    public SaveInstructTemplateBuilder setPourSaveLogic(PourSaveLogic pourSaveLogic) {
        this.pourSaveLogic = pourSaveLogic;
        return this;
    }
}
