package com.aote.util.other;


import com.aote.config.SystemConfig;
import com.aote.util.ParamFilter;
import com.aote.util.Util;
import org.json.JSONObject;
import org.springframework.util.StringUtils;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.spec.SecretKeySpec;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;

/**
 * AES的加密和解密
 */
public class AESUtil {
    //算法
    private static final String ALGORITHMSTR = "AES/ECB/PKCS5Padding";

    /**
     * 加密方法
     * @param content 待加密的内容
     * @param encryptKey 加密密钥
     * @return 加密后的字符串
     */
    public static String encrypt(String content, String encryptKey) {
        try {
            return aesEncrypt(content, encryptKey);
        } catch (Exception e) {
            e.printStackTrace();
            return "";
        }
    }

    /**
     * 解密方法
     * @param content 待解密的内容
     * @param encryptKey 解密密钥
     * @return 解密后的字符串
     */
    public static String decrypt(String content, String encryptKey) {
        try {
            return aesDecrypt(content, encryptKey);
        } catch (Exception e) {
            e.printStackTrace();
            return "";
        }
    }

    /**
     * 将byte[]转为各种进制的字符串
     * @param bytes byte[]
     * @param radix 可以转换进制的范围，从Character.MIN_RADIX到Character.MAX_RADIX，超出范围后变为10进制
     * @return 转换后的字符串
     */
    public static String binary(byte[] bytes, int radix){
        return new BigInteger(1, bytes).toString(radix);// 这里的1代表正数
    }

    /**
     * AES加密
     * @param content 待加密的内容
     * @param encryptKey 加密密钥
     * @return 加密后的byte[]
     * @throws Exception
     */
    public static byte[] aesEncryptToBytes(String content, String encryptKey) throws Exception {
        KeyGenerator kgen = KeyGenerator.getInstance("AES");
        kgen.init(128);
        Cipher cipher = Cipher.getInstance(ALGORITHMSTR);
        cipher.init(Cipher.ENCRYPT_MODE, new SecretKeySpec(encryptKey.getBytes(), "AES"));

        return cipher.doFinal(content.getBytes(StandardCharsets.UTF_8));
    }

    /**
     * AES加密为base 64 code
     * @param content 待加密的内容
     * @param encryptKey 加密密钥
     * @return 加密后的base 64 code
     * @throws Exception
     */
    public static String aesEncrypt(String content, String encryptKey) throws Exception {
        return Base64Utils.encode(aesEncryptToBytes(content, encryptKey));
    }

    /**
     * AES解密
     * @param encryptBytes 待解密的byte[]
     * @param decryptKey 解密密钥
     * @return 解密后的String
     * @throws Exception
     */
    public static String aesDecryptByBytes(byte[] encryptBytes, String decryptKey) throws Exception {
        KeyGenerator kgen = KeyGenerator.getInstance("AES");
        kgen.init(128);

        Cipher cipher = Cipher.getInstance(ALGORITHMSTR);
        cipher.init(Cipher.DECRYPT_MODE, new SecretKeySpec(decryptKey.getBytes(), "AES"));
        byte[] decryptBytes = cipher.doFinal(encryptBytes);
        return new String(decryptBytes, StandardCharsets.UTF_8);
    }

    /**
     * 将base 64 code AES解密
     * @param encryptStr 待解密的base 64 code
     * @param decryptKey 解密密钥
     * @return 解密后的string
     * @throws Exception
     */
    public static String aesDecrypt(String encryptStr, String decryptKey) throws Exception {
        return StringUtils.isEmpty(encryptStr) ? null : aesDecryptByBytes(Base64Utils.decodeBuffer(encryptStr), decryptKey);
    }

    /**
     * 从后台config.json配置加密
     * */
    public static String addDecrypt(String paramStr){
        // 数据加密
//       configJson=={"proxy":"http://127.0.0.1:8300","ldap":"http://127.0.0.1:8300","encryKey":"OXuYieBb4eoIne^K","isEncrypt":true}
        if (SystemConfig.DataSecurity.getEnableDataEncryption()
            && SystemConfig.DataSecurity.hasAesKey()) {
            String encryKey = SystemConfig.DataSecurity.getAesKey();
            return AESUtil.encrypt(paramStr, encryKey);
        }else{
//            System.out.println("未加密--"+paramStr);
            return paramStr;
        }
    }
    public static String getEncrypt(String str){
        // 数据解密
        if (SystemConfig.DataSecurity.getEnableDataEncryption()
                && SystemConfig.DataSecurity.hasAesKey()) {
            String encryKey = SystemConfig.DataSecurity.getAesKey();
            return decrypt(str, encryKey);
        }else{
            return str;
        }
    }

    @FunctionalInterface
    public interface AfterFunction {
        /**
         * 解密回调
         * @param data 根据加密类型返回结果
         * @return 解密回调的返回
         */
        String run(String data) throws Exception;
    }

    /**
     * 加解密业务
     * @param paramJson 参数JSON
     * @param defaultFunction 解密后回调
     * @return
     * @throws Exception
     */
    public static String encryptData(JSONObject paramJson,
                                 AfterFunction defaultFunction) throws Exception {
        //验证
        Util.checkEncry(paramJson);
        //如果不需要加密，进入default正常执行
        if (!paramJson.has("aoteEncrypt")){
            paramJson.put("aoteEncrypt", "default");
        }
        //sql注入标志位
        boolean sqlInject = SystemConfig.DataSecurity.getInjectionCheck();
        //根据解密方式进行解密
        if ("AES".equals(paramJson.getString("aoteEncrypt"))) {
            if (!SystemConfig.DataSecurity.hasAesKey()) {
                throw new RuntimeException("未在" + SystemConfig.CONFIG_PATH + "中配置AES密钥信息！");
            }
            // 数据解密
            String encryKey = SystemConfig.DataSecurity.getAesKey();
            String data = paramJson.getString("data");
            String decryData = AESUtil.decrypt(data, encryKey);
            //验证参数是否合法
            if (sqlInject) {
                ParamFilter.checkSqlJsonStr(decryData);
            }
            return AESUtil.encrypt(defaultFunction.run(decryData), encryKey);
        }
        String paramData = paramJson.toString();
        //验证参数是否合法
        if (sqlInject) {
            ParamFilter.checkSqlJsonStr(paramData);
        }
        return defaultFunction.run(paramData);
    }
}
