package com.aote.util;

import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.StrUtil;
import com.aote.logic.LogicServer;
import com.aote.sql.SqlServer;
import com.aote.transaction.SessionPool;
import com.aote.transaction.ThreadSession;
import org.apache.log4j.Logger;
import org.hibernate.SessionFactory;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Iterator;

/**
 * @Author: JoeCdy
 * @Description: 单表查询工具类
 * @Date: Created in 9:28 2022/7/26
 */
@Component
@Transactional
public class SingleTableTool {

    static Logger log = Logger.getLogger(SingleTableTool.class);

    @Autowired
    private SqlServer sqlServer;

    @Autowired
    private LogicServer logicServer;

    @Autowired
    private ThreadSession tSession;

    @Autowired
    private SessionPool sessionPool;

    @Autowired
    private SessionFactory factory;


    // 生成临时表名
    public static String temporaryName() {
        String name = "LS_" + IdUtil.objectId();
        // oracle防止出现大小写问题, 表名全部转为大写
        name = name.toUpperCase();
        return name;
    }

    /**
     * 生成指定数量的临时表名称
     * @param num 需要数量
     * @return 生成的临时表列表
     */
    public static ArrayList<String> temporaryName (int num) {
        ArrayList<String> ls = new ArrayList<>();
        for (int i = 0; i < num; i++) {
            ls.add(temporaryName());
        }
        return ls;
    }

    // 生成一个占位符, 解决logic中{}语法冲突
    public static String getPlaceholder() {
        return "{}";
    }

    /**
     * 创建一个临时表
     *
     * @param logicConfig 返回配置的logic名称
     * @param keyName     指定配置的key
     * @param querySql    用于插入临时表数据的查询sql
     * @return 临时表名称
     */
    public String createTemporaryTable(String logicConfig, String keyName, String querySql) throws Exception {
        return createTemporaryTable(logicConfig, keyName, querySql, null);
    }

    /**
     * 创建一个临时表(指定表名)
     *
     * @param logicConfig 返回配置的logic名称
     * @param keyName     指定配置的key
     * @param querySql    用于插入临时表数据的查询sql
     * @param tableName   指定生成的临时表表名
     * @return 临时表名称
     */
    public String createTemporaryTable(String logicConfig, String keyName, String querySql,String tableName) throws Exception {
        try {
            if (tableName == null) {
                tableName = temporaryName();
            }
            // 获取配置
            JSONObject config = (JSONObject) logicServer.run(logicConfig, new JSONObject());
            JSONArray json = config.getJSONArray(keyName);
            StringBuilder fields = new StringBuilder();
            StringBuilder fieldsInto = new StringBuilder();
            // 生成临时表的字段
            for (int i = 0; i < json.length(); i++) {
                AbstractMap.SimpleEntry<String, JSONObject> row = (AbstractMap.SimpleEntry) json.get(i);
                String field = row.getKey();
                JSONObject value = row.getValue();
                String defaultType = "VARCHAR2(100)";
                if (value.has("type")) {
                    defaultType = value.getString("type");
                }
                fields.append(StrUtil.format("{} {},", field, defaultType));
                fieldsInto.append(StrUtil.format("{},", field));
            }
            // 用于创建的字段字符串
            String fieldStr = StrUtil.removeSuffix(fields, ",");
            // 用于插入的字段字符串
            String fieldsIntoStr = StrUtil.removeSuffix(fieldsInto, ",");
            // oracle创建临时表语句模板
            String createSqlTem = "CREATE GLOBAL TEMPORARY TABLE {} (\n{}) ON COMMIT PRESERVE ROWS";
            // 最终生成的建表语句
            String createSql = StrUtil.format(createSqlTem, tableName, fieldStr);
            // 执行建表语句
            int i = sqlServer.runSQL(createSql);
            // 插入语句模板
            String insertSqlTem = "insert into {} ({}) {}";
            // 最终生成的插入语句
            String insertSql = StrUtil.format(insertSqlTem, tableName, fieldsIntoStr, querySql);
            int i1 = sqlServer.runSQL(insertSql);
            log.debug("执行INSERT语句结果:" + i1);
            return tableName;
        } catch (Exception e) {
            log.debug("创建一个临时表异常:", e);
            throw e;
        }
    }

    /**
     * 根据配置生成每个单表的查询条件
     *
     * @param logicConfig 配置logic名称
     * @param model       参数model
     * @return 每个表对应条件
     * @throws Exception
     */
    public JSONObject generateConditions(String logicConfig, JSONObject model) throws Exception {
        try {
            JSONObject result = new JSONObject();
            // 获取配置
            JSONObject config = (JSONObject) logicServer.run(logicConfig, new JSONObject());
            Iterator<String> keys = config.keys();
            while (keys.hasNext()) {
                String next = keys.next();
                JSONArray json = config.getJSONArray(next);
                StringBuilder condition = new StringBuilder("1=1");
                for (int i = 0; i < json.length(); i++) {
                    AbstractMap.SimpleEntry<String, JSONObject> row = (AbstractMap.SimpleEntry) json.get(i);
                    String field = row.getKey();
                    if (model.has(field) && !"".equals(model.get(field))) {
                        JSONObject value = row.getValue();
                        // 生成条件方式
                        String joint = value.optString("joint", "= '{}'");
                        if ("false".equals(joint)) {
                            continue;
                        }
                        condition.append(StrUtil.format(" and {} " + joint, field, model.getString(field)));
                    }
                }
                result.put(next, condition);
            }
            return result;
        } catch (Exception e) {
            log.debug("生成查询条件异常:", e);
            throw e;
        }
    }
}
