package com.aote.rs;

import com.af.plugins.ConvertTools;
import com.aote.exception.ExceptionLogManage;
import com.aote.logic.LogicMapper;
import com.aote.logic.LogicServer;
import com.aote.mq.RocketMQProducer;
import com.aote.redis.RedisUtil;
import com.aote.rs.mapper.WebException;
import com.aote.util.ExceptionHelper;
import com.aote.util.ResourceHelper;
import com.aote.util.SnowflakeIdWorker;
import org.apache.log4j.Logger;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.common.message.Message;
import org.apache.rocketmq.remoting.exception.RemotingException;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import javax.annotation.PostConstruct;
import javax.inject.Singleton;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;

@Path("logic")
@Singleton
@Component
public class LogicService {
	static Logger log = Logger.getLogger(LogicService.class);
	static Logger reviewLog = Logger.getLogger("reviewLog");

	@Autowired
	private LogicServer logicServer;

	@Autowired
	private RocketMQProducer producer;

	@Autowired
	private ResourceHelper resourceHelper;

	@Autowired
	private ExceptionLogManage exceptionLogManage;

	public static final ThreadLocal<String> reviewKey = new ThreadLocal<>();
	public static final Map<String, JSONArray> reviewCache = new ConcurrentHashMap<>();

	/**
	 * 清理Logic源缓存
	 * @return ok
	 */
	@GET
	@Path("/clear")
	public String clearLogicCache(){
		ResourceHelper.clearLogicCache();
		return "ok";
	}

	/**
	 * 清理所有缓存
	 * @return ok
	 */
	@GET
	@Path("/clearall")
	public String clearAllCache(){
		ResourceHelper.clearLogicCache();
		ResourceHelper.clearSqlCache();
		return "ok";
	}

	/**
	 * 执行业务逻辑(GET方式)
	 *
	 * @param logicName 业务逻辑名称
	 * @param request   HttpServletRequest对象
	 * @return 执行结果
	 */
	@GET
	@Path("{logic}")
	public Response xtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) {
		String requestParams = getRequestParams(null, request);
		String requestId = request.getHeader("X-Request-ID");
		return runMain(logicName, requestParams, requestId);
	}

	/**
	 * 执行业务逻辑(POST方式)
	 *
	 * @param logicName 业务逻辑名称
	 * @param values JSON资源字符串
	 * @return 执行结果
	 */
	@POST
	@Path("{logic}")
	public Response xtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request) {
		String requestParams = getRequestParams(values, request);
		String requestId = null;
		if (request != null) {
			requestId = request.getHeader("X-Request-ID");
		}
		return runMain(logicName, requestParams, requestId);
	}

	@GET
	@Path("mq/{logic}")
	public Response mqXtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) throws Exception {
		Map paramMap = request.getParameterMap();
		JSONObject params = new JSONObject(paramMap);
		String values = params.toString();
		return runMQ(logicName, values);
	}

	@POST
	@Path("mq/{logic}")
	public Response mqXtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request) throws Exception {
		String requestParams = getRequestParams(values, request);
		return runMQ(logicName, requestParams);
	}

	/**
	 * 异步执行业务逻辑(GET方式)
	 * https://stackoverflow.com/questions/31401640/spring-xml-equivalent-of-enableasync
	 *
	 * @param logicName 业务逻辑名称
	 * @param request   HttpServletRequest对象
	 * @return 执行结果
	 */
	@GET
	@Path("async/{logic}")
	public Response asyncXtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) {
		Map<String, String[]> paramMap = request.getParameterMap();
		JSONObject params = new JSONObject(paramMap);
		try {
			logicServer.runAsync(logicName, params.toString());
		} catch (Exception ignore){}
		//移动平台推送特殊处理
		if(params.has("msg")){
			return Response.ok(params.getJSONArray("msg").get(0)).build();
		}
		return Response.ok(new JSONObject().put("returnCode", "0000").toString()).build();
	}

	/**
	 * 异步执行业务逻辑(POST方式)
	 * https://stackoverflow.com/questions/31401640/spring-xml-equivalent-of-enableasync
	 *
	 * @param logicName 业务逻辑名称
	 * @param values JSON资源字符串
	 * @return 执行结果
	 */
	@POST
	@Path("async/{logic}")
	public Response asyncXtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request) {
		try {
			logicServer.runAsync(logicName, values);
		} catch (Exception ignore){}
		return Response.ok(new JSONObject().put("returnCode", "0000").toString()).build();
	}

	public String xtSave(String logicName, String values) {
		return xtSave(logicName,values,null).getEntity().toString();
	}

	private Response runMain(String logicName, String values, String requestId){
		String id = requestId != null ? requestId : SnowflakeIdWorker.getValue().toString();
		reviewKey.set(id);
		long begin = System.currentTimeMillis();
		JSONObject reviewObj = new JSONObject();
		reviewObj.put("requestId", id);
		reviewCache.put(id, new JSONArray());
		RedisUtil redisUtil = RedisUtil.getInstance();
		try {
			reviewObj.put("status", 0);
			boolean isEnableCache = LogicMapper.isEnableCache(logicName);
			if (!isEnableCache){
				Object result = logicServer.run(logicName, values);
				return buildResponse(result);
			}
			// 获取自定义缓存Key
			String cacheKeyValue = LogicMapper.getCacheKey(logicName);
			String paramsStr;
			if (!values.startsWith("{")){
				log.warn("Logic["+logicName+"]的请求参数不满足获取自定义缓存Key，已跳过获取缓存");
				Object result = logicServer.run(logicName, values);
				return buildResponse(result);
			}
			JSONObject params = new JSONObject(values);
			if (Optional.ofNullable(cacheKeyValue).isPresent() && params.has(cacheKeyValue)) {
				// 如果调用Logic的参数中含有该key，则取该key对应的值，否则直接以参数列表作为缓存key
				paramsStr = params.get(cacheKeyValue).toString();
			} else {
				paramsStr = ConvertTools.base64Encode(new TreeMap<>(params.toMap()).toString().getBytes(StandardCharsets.UTF_8));
			}
			String cacheKey = "LogicDataCache@" + logicName + "-" + paramsStr;
			Object cacheResult = redisUtil.get(cacheKey);
			if (cacheResult != null){
				log.debug("维护信息:Logic["+logicName+"]命中缓存，参数：" + values);
				return buildResponse(cacheResult.toString());
			}
			try {
				long cacheTime = LogicMapper.getCacheTime(logicName);
				Object result = redisUtil.lock(cacheKey, cacheTime <= 0 ? 0 : 60, 300, ()->{
					if (cacheTime <= 0) {
						return logicServer.run(logicName, values);
					}
					Object newCacheResult = redisUtil.get(cacheKey);
					if (newCacheResult != null) {
						log.debug("维护信息:Logic["+logicName+"]命中缓存，参数：" + values);
						return newCacheResult;
					}
					newCacheResult = logicServer.run(logicName, values);
					if(newCacheResult != null){
						redisUtil.set(cacheKey, newCacheResult, cacheTime);
						log.debug("维护信息:Logic["+logicName+"]缓存成功，存活时间：" + cacheTime + "秒");
					}
					return newCacheResult;
				});
				return buildResponse(result.toString());
			} catch (InterruptedException e) {
				String demoteLogicName = LogicMapper.getDemoteLogicName(logicName);
				if (demoteLogicName != null) {
					//由于锁占用，业务降级处理
					return buildResponse(logicServer.run(demoteLogicName, values));
				} else {
					return Response.status(Response.Status.CONFLICT).build();
				}
			}
		} catch (Exception e){
			// 存储异常日志
			String stack = ExceptionLogManage.getErrorInfo(ExceptionHelper.stackToString(e),1);
			log.error(stack);
			// 获取源程序路径
			String path = LogicMapper.getLogic(logicName);
			if(path == null){
				path = logicName;
			}
			exceptionLogManage.writeError(e,path,"/rs/logic/"+logicName,values);
			reviewObj.put("status",-1);
			WebException wex = this.getWebException(e);
			if (wex != null)
			{
				throw wex;
			}
			throw e;
		} finally {
			long end = System.currentTimeMillis();
			reviewObj.put("bTime", begin);
			reviewObj.put("eTime", end);
			reviewObj.put("constTime", end - begin);
			reviewObj.put("type", "logic");
			reviewObj.put("items", reviewCache.get(id));
			reviewLog.debug(reviewObj);
			reviewKey.remove();
			reviewCache.remove(id);
		}
	}

	private Response runMQ(String logicName, String values) throws RemotingException, InterruptedException, MQClientException {
		Message msg = new Message("logic",logicName,values.getBytes(StandardCharsets.UTF_8));
		// 发送单向消息
		producer.getDefaultMQProducer().sendOneway(msg);
		//移动平台推送特殊处理
		if(values.startsWith("{")){
			JSONObject params = new JSONObject(values);
			if(params.has("msg")) {
				return Response.ok(params.getJSONArray("msg").get(0)).build();
			}
		}
		return Response.ok().build();
	}

	private Response buildResponse(Object metaResult) {
		String result;
		if (metaResult == null) {
			result = "";
		} else if (metaResult instanceof Map<?, ?>) {
			result = new JSONObject((Map<?, ?>) metaResult).toString();
		} else if (metaResult instanceof Collection<?>) {
			result = new JSONArray((Collection<?>) metaResult).toString();
		} else {
			result = metaResult.toString();
		}
		Response.ResponseBuilder builder = Response.ok();
		if (result.startsWith("{")) {
			JSONObject paramJson = new JSONObject(result);
			//如果Logic返回值中包含header和content关键字，则认为需要给调用方指定的响应头
			if (paramJson.has("header") && paramJson.has("content")) {
				Map<String, Object> paramHeader = paramJson.getJSONObject("header").toMap();

				for (Map.Entry<String, Object> stringObjectEntry : paramHeader.entrySet()) {
					builder.header(stringObjectEntry.getKey(), String.valueOf(stringObjectEntry.getValue()));
				}
				return builder.type("text/plain").entity(String.valueOf(paramJson.get("content"))).build();
			} else {
				builder.type(MediaType.APPLICATION_JSON_TYPE);
			}
		}

		return builder.entity(result).build();
	}

	//服务器启动时加载模块至redis
	@PostConstruct
	public void redisCacheLogic() {
		logicServer.redisCacheLogic();
	}

	// 找到WebException异常
	// return - 不存在，返回null
	private WebException getWebException(Exception ex) {
		Throwable cause = ex;
		while (cause != null)
		{
			if (cause instanceof WebException)
			{
				return (WebException)cause;
			}
			cause = cause.getCause();
		}
		return null;
	}

    /**
     * 获取请求参数
     *
     * @param values  请求体内容
     * @param request HttpServletRequest
     * @return 请求参数
     */
    private String getRequestParams(String values, HttpServletRequest request) {
        JSONObject paramsJson;
        if (StringUtils.isEmpty(values)) {
            paramsJson = new JSONObject();
        } else if (values.startsWith("{") && values.endsWith("}")) {
            paramsJson = new JSONObject(values);
        } else {
            paramsJson = new JSONObject();
            paramsJson.put("str", values);
            //加入请求头数据
            Enumeration<String> headerNames = request.getHeaderNames();
            if (headerNames != null) {
                JSONObject header = new JSONObject();
                while (headerNames.hasMoreElements()) {
                    String temp = headerNames.nextElement();
                    header.put(temp, request.getHeader(temp));
                }
                paramsJson.put("header", header);
            }
        }
		if (request != null) {
			//加入请求地址参数
			Map<String, String[]> parameterMap = request.getParameterMap();
			parameterMap.forEach((key, value) -> {
				if (value.length > 0 && !paramsJson.has(key)) {
					paramsJson.put(key, value[0]);
				}
			});
		}
		return paramsJson.toString();
    }
}
