package com.aote.sql;

import com.af.util.Pair;
import com.aote.entity.EntityServer;
import com.aote.exception.FileNotFoundException;
import com.aote.transaction.SessionPool;
import com.aote.util.*;
import org.apache.log4j.Logger;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.transaction.Transactional;
import java.io.File;
import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.*;

@Component
@Transactional
public class SqlServer {
	static Logger log = Logger.getLogger(SqlServer.class);

	@Autowired
	public SessionFactory sessionFactory;

	@Autowired
	private SessionPool sessionPool;

	@Autowired
	public EntityServer entityServer;

	public SessionFactory getSessionFactory() {
		return sessionFactory;
	}

	public Session assignedSession;

	public void setAssignedSession(Session session) {
		this.assignedSession = session;
	}


	/**
	 * 获取SQL语句的合计执行结果
	 *
	 * @param name
	 *            : sql语句名
	 * @param str
	 *            : sql语句执行参数
	 * @return JSON格式Sql语句执行结果
	 * @throws JSONException
	 */
	public JSONObject queryTotal(String name, String str) throws Exception {

		// 获取参数，求和字段等内容
		JSONObject param = null;
		Object sums = null;
		if (str != null && !str.isEmpty()) {
			JSONObject json = new JSONObject(str);
			if (json.has("data")) {
				param = json.getJSONObject("data");
			} else {
				param = new JSONObject();
			}
			if (json.has("sums")) {
				sums = json.get("sums");
			}
		} else {
			param = new JSONObject();
		}
		Map<String, Object> params = JsonHelper.toMap(param);
		// 产生sql语句编译后的结果
		String sql = this.call(name, params);

		// 求和时，order by会导致sql错误，过滤掉order by部分。
		sql = filterOutOrderBy(sql, sums);
		Session session = (assignedSession == null ? sessionPool.getSession() : assignedSession);
		JSONArray array = SqlHelper.query(session, sql);
		return array.getJSONObject(0);
	}

	/**
	 * 获取HQL语句的合计执行结果
	 *
	 * @param name
	 *            : hql语句名
	 * @param str
	 *            : hql语句执行参数
	 * @return JSON格式hql语句执行结果
	 * @throws JSONException
	 */
	public JSONObject queryHqlTotal(String name, String str) throws Exception {
		// 获取参数，求和字段等内容
		JSONObject param;
		if (str != null && !str.isEmpty()) {
			JSONObject json = new JSONObject(str);
			if (json.has("data")) {
				param = json.getJSONObject("data");
			} else {
				param = new JSONObject();
			}
		} else {
			param = new JSONObject();
		}
		Map<String, Object> params = param.toMap();
		// 产生sql语句编译后的结果
		String sql = this.call(name, params);
		Session session = (assignedSession==null ? sessionPool.getSession() : assignedSession);
		JSONArray array = SqlHelper.hqlQuery(session, sql);
		return array.getJSONObject(0);
	}

	/**
	 * 执行sql分页查询
	 */
	public JSONArray query(String name, String str) throws Exception {
		return query(name, 0, 1000, str);
	}

	/**
	 * 删除文件
	 *
	 * @param filePath 文件路径
	 */
	public static boolean deleteFile(String filePath) {
		File file = new File(filePath);
		if (file.isFile() && file.exists()) {
			boolean succeedDelete = file.delete();
			if (succeedDelete) {
				log.debug("删除" + filePath + "成功！");
			} else {
				log.debug("删除" + filePath + "失败！");
			}
			return true;
		} else {
			log.debug("删除" + filePath + "失败！");
			return false;
		}
	}

	// 参数为json对象
	public JSONArray query(String name, JSONObject params) throws Exception {
		Map<String, Object> map = JsonHelper.toMap(params);
		return query(name, 1, 1000, map);
	}

	public JSONArray query(String name, JSONObject params, Integer pageSize) throws Exception {
		Map<String, Object> map = JsonHelper.toMap(params);
		return pageSize <= 0 ? query(name, 1, 9999999, map) : query(name, 1, pageSize, map);
	}

	// 参数为map
	public JSONArray query(String name, Map<String, Object> params)
			throws Exception {
		return query(name, 1, 1000, params);
	}

	public JSONArray query(String name, Map<String, Object> params, Integer pageSize)
			throws Exception {
		return pageSize <= 0 ? query(name, 1, 9999999, params) : query(name, 1, pageSize, params);
	}

	public JSONArray query(String name, int pageNo, int pageSize, String str)
			throws Exception {
		// pageNo小于0， 纠正成1
		if (pageNo <= 0) {
			pageNo = 1;
		}

		// pageSize小于0，纠正成1, 解除1000限制
		if (pageSize < 1) {
			pageSize = 9999999;
		}

		// 拿到json对象参数
		JSONObject param;
		if (str != null && !str.isEmpty()) {
			param = new JSONObject(str);
			param = param.getJSONObject("data");
		} else {
			param = new JSONObject();
		}
		Map<String, Object> params = param.toMap();
		return query(name, pageNo, pageSize, params);
	}

	public JSONArray query(String name, int pageNo, int pageSize,
			Map<String, Object> params) throws Exception {
		// 产生SQL语句编译后的结果
		String sql = this.call(name, params);

		Session session = ( assignedSession == null ? sessionPool.getSession() : assignedSession);
		JSONArray array = SqlHelper.query(session, sql, pageNo - 1, pageSize);
		log.info("SQL["+ name+"] Result: \n"+ array.toString());
		return array;
	}

	/**
	 * 获取SQL文件的编译后SQL
	 * @param name SQL文件映射名
	 * @param params 参数
	 * @return SQL语句
	 */
	public String getSQL(String name, JSONObject params){
		//转为Map
		Map<String, Object> map = JsonHelper.toMap(params);
		return this.call(name, map);
	}

	@Deprecated
	public JSONArray query(String sql) throws Exception{
		return querySQL(sql);
	}

	public JSONArray querySQL(String sql) throws Exception{
		return querySQL(sql, 1, 9999999, false);
	}

	public JSONArray querySQL(String sql, boolean isNoLog) throws Exception{
		return querySQL(sql, 1, 9999999, isNoLog);
	}

	public JSONArray querySQL(String sql,Integer pageSize) throws Exception {
		return querySQL(sql, pageSize, false);
	}

	public JSONArray querySQL(String sql,Integer pageSize, boolean isNoLog) throws Exception{
		return pageSize <= 0 ? querySQL(sql, 1, 9999999,isNoLog) : querySQL(sql, 1, pageSize,isNoLog);
	}

	public JSONArray querySQL(String sql,int pageNo, int pageSize) throws Exception {
		return querySQL(sql, pageNo, pageSize, false);
	}

	public JSONArray querySQL(String sql,int pageNo, int pageSize, boolean isNoLog) throws Exception {
		if (pageNo <= 0) {
			pageNo = 1;
		}

		if (pageSize < 1) {
			pageSize = 9999999;
		}
		Session session = (assignedSession==null ? sessionPool.getSession() : assignedSession);
		JSONArray array = SqlHelper.query(session, sql, pageNo - 1, pageSize, isNoLog);
		if(!isNoLog) {
			log.info("SQL Result: \n"+ array.toString());
		}
		return array;
	}

	// 执行sql语句
	public void run(String sql) throws NullPointerException {
		Session session = (assignedSession==null ? sessionPool.getSession() : assignedSession);
		SqlHelper.bulkSQLUpdate(session, sql);
	}

	public void run(String sql,Object... params){
		String useSql = String.format(sql.replace("{}","%s"),params);
		Session session = (assignedSession==null ? sessionPool.getSession() : assignedSession);
		SqlHelper.bulkSQLUpdate(session, useSql);
	}

	/**
	 * 执行删除sql语句
	 * @param session  会话
	 * @param sql  删除语句
	 */
	public void runDeleteSQL(Session session,String sql) {
		EntityServer entityServer=new EntityServer();
		String[] sqlStr=sql.trim().split("\\s+");
		List<String> childStr =entityServer.getChildName(sqlStr[2]);
		//如果有子表，先执行子表语句
		for (String str : childStr) {
			String sqlRes=sql.replaceFirst(sqlStr[2], str);
			doCascade(session,sqlRes);
			SqlHelper.bulkSQLUpdate(session, sqlRes);//先删除其继承关系
		}
		doCascade(session,sql);
		SqlHelper.bulkSQLUpdate(session, sql);//删除自己关系
	}
	/**
	 * 执行更新sql语句
	 * @param session  会话
	 * @param sql  更新语句
	 */
	public void runUpdateSQL(Session session,String sql) {
		EntityServer entityServer=new EntityServer();
		String[] sqlStr=sql.trim().split("\\s+");
		List<String> childStr =entityServer.getChildName(sqlStr[1]);
		//如果有子表，先执行子表语句
		for (String str : childStr) {
			String sqlRes=sql.replaceFirst(sqlStr[1], str);
			SqlHelper.bulkSQLUpdate(session, sqlRes);//先更新其继承关系
		}
		SqlHelper.bulkSQLUpdate(session, sql);//先更新自己关系
	}

	/**
	 * 执行查询SQL语句
	 * @param sql SQL语句
	 * @return ResultSet对象
	 */
	public ResultSet runQuerySQL(String sql){
		Session session = (assignedSession == null ? sessionPool.getSession() : assignedSession);
		return session.doReturningWork(
				connection -> {
					PreparedStatement preparedStatement=connection.prepareStatement(sql);
					return preparedStatement.executeQuery();
				}
		);
	}

	private static List<Map<String, Object>> saveList(JSONArray array) throws Exception {
		List<Map<String, Object>> set = new ArrayList<>();

		for (int i = 0; i < array.length(); ++i) {
			JSONObject obj = (JSONObject) array.get(i);
			Map<String, Object> map = toMap(obj);
			set.add(map);
		}

		return set;
	}

	private static Map<String, Object> toMap(JSONObject object) {
		return object.toMap();
	}

	public JSONArray query(String name, String str, Integer pageSize) throws Exception {
		return pageSize <= 0 ? query(name, 1, 9999999, str) : query(name, 1, pageSize, str);
	}

	/**
	 * 获取级联删除语句的条件
	 * @param session 会话
	 * @param sql 级联删除语句
	 */
	public void doCascade(Session session, String sql){
		sql = sql.trim();          //去除空格
		HashMap<String, HashMap<String, Object>> metaMap = EntityServer.metaMap;//得到实例map
		String[] sqlStr=sql.split("\\s+");
		String idnum=(String) metaMap.get(sqlStr[2]).get("idColName"); //获取删除表名的主键名字
		String resql=sql.replaceFirst(sqlStr[0], "SELECT "+idnum); // 替换成select语句用来得到主键值value
		try {
			JSONArray value = querySQL(resql);          //获取到查询条件
			if(value.length() > 0) {
				for (int j = 0; j < value.length(); j++) {
					JSONObject job = value.getJSONObject(j);
					String res=job.get(idnum).toString();
					cascadeDelete(session,sqlStr[2], res);//sqlStr[2]：表名；res：主键值
				}
			}
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

	// 执行sql语句，非hql 返回受影响行数
	public int runSQL(String sql) {
		Session session = (assignedSession == null ? sessionPool.getSession() : assignedSession);
		return SqlHelper.bulkSQLUpdate(session, sql);
	}

	public int runSQL(String sql,Object... params){
		String useSql = String.format(sql.replace("{}","%s"),params);
		Session session = (assignedSession==null ? sessionPool.getSession() : assignedSession);
		return SqlHelper.bulkSQLUpdate(session, useSql);
	}

	// 基于JSON对象的call函数
	public String call(String sqlName, JSONObject json) throws Exception {

		Map<String, Object> param = JsonHelper.toMap(json);
		return call(sqlName, param);
	}

	// 调用其它sql语句, 产生sql语句经过参数处理的结果串
	public String call(String sqlName, Map<String, Object> params) {
		// 获取原始sql语句
		String path = SqlMapper.getSql(sqlName);
		if (path == null) {
			throw new RuntimeException("sql语句未注册！" + sqlName);
		}
		String sql;
		try {
			sql = (String) ResourceHelper.getString(ResourceType.SQL,sqlName,path);
			//如果使用了新的格式化参数写法，需要替换为解析语法树支持的写法
			if(sql.contains("${")){
				sql = sql.replace("${","{");
			}
		} catch (FileNotFoundException e) {
			throw new RuntimeException(path + ".文件无配置");
		} catch (IOException io) {
			throw new RuntimeException(io);
		}
		// 把自身注册到执行环境中
		params.put("sql", this);
		params.put("entity", entityServer);
		sql = ExpressionHelper.run(sql, params).toString();
		return sql;
	}

	/**
	 * 级联删除循环体，用于执行每一个删除语句
	 * @param session  会话
	 * @param tableName 删除表名
	 * @param idValue   删除表主键值
	 */
public void cascadeDelete(Session session,String tableName,String idValue) {
	HashMap<String, HashMap<String, Object>> metaMap = EntityServer.metaMap; // 得到实例map
		String idNum = (String)metaMap.get(tableName).get("idColName");   // 获取删除表名的主键名字
		//执行判断是否有path属性来删除文件
		deleteFileDB(tableName, idValue);
		//继续递归自己，深度级联
		if(metaMap.containsKey(tableName)){
			List<String> cascadeTable = new ArrayList<String>();
			HashMap<String, Pair> cascadMap=(HashMap<String, Pair>) metaMap.get(tableName).get("onetoone");
			cascadMap.putAll((HashMap<String, Pair>) metaMap.get(tableName).get("onetomany"));
			Set<String> keys = cascadMap.keySet();// 得到全部的也就是子表属性名字
			Iterator<String> iter = keys.iterator();
			int i = 0;
			//获取所有级联属性
			while (iter.hasNext()) {
				cascadeTable.add(iter.next());
				i++;
			}
			//pair.type:外键
			//pair.col：表名
			for (String str : cascadeTable) {
				Pair pair = cascadMap.get(str);
				String tableString = (String) metaMap.get(pair.col).get("tableName");          //获取表名
				String tableId = (String) metaMap.get(pair.col).get("idColName");                //获取主键名
				String selSql;
				//判断id是否为String型
				if (metaMap.get(pair.col).get("idType") == "STRING") {
					selSql = "SELECT " + tableId + " FROM " + tableString + " WHERE " + pair.type + "='" + idValue + "'";  //拼接查询id的sql语句
				} else {
					selSql = "SELECT " + tableId + " FROM " + tableString + " WHERE " + pair.type + "=" + idValue;  //拼接查询id的sql语句
				}
				//sql语句，得到所有外键的值
				try {
					JSONArray array = querySQL(selSql);  //获取到查询条件
					if (array.length() > 0) {
						for (int j = 0; j < array.length(); j++) {
							JSONObject job = array.getJSONObject(j);
							String res = job.get(tableId).toString();
							String sqlStr;
							if (job.get(tableId) instanceof String) {
								sqlStr = "DELETE FROM " + pair.col + " WHERE " + tableId + "='" + res + "'";    //执行级联删除的主要语句
							} else {
								sqlStr = "DELETE FROM " + pair.col + " WHERE " + tableId + "=" + res;    //执行级联删除的主要语句
							}
							runDeleteSQL(session, sqlStr);   //递归调用自身删除
						}
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	String delSql;
		if(metaMap.get(tableName).get("idType") == "STRING") {
			delSql = "DELETE FROM " + tableName + " WHERE " + idNum + "= '" + idValue + "'";
		} else {
			delSql = "DELETE FROM " + tableName + " WHERE " + idNum +"="+idValue;
		}
		//执行级联删除的主要语句
		SqlHelper.bulkSQLUpdate(session, delSql);
	}

	/**
	 * 判断是否有path属性，查询所要删除的文件路径，并执行删除
	 * @param tableName   表名
	 * @param idValue     表主键值
	 */
	public void deleteFileDB(String tableName,String idValue){
		HashMap<String, HashMap<String, Object>> metaMap = EntityServer.metaMap; // 得到实例map
		Map<String, Object> columns = (Map<String, Object>)metaMap.get(tableName).get("columns");
		if (columns != null && columns.containsKey("path")){
			String idNum = (String)metaMap.get(tableName).get("idColName");   // 获取删除表名的主键名字
			String selSql="SELECT path FROM "+tableName+" WHERE "+idNum+"="+idValue;  //拼接查询id的sql语句
			try {
				JSONArray array = querySQL(selSql);   //获取到查询条件
				if(array.length()>0){
					for(int i=0;i<array.length();i++){
						JSONObject job = array.getJSONObject(i);
						String filePath=job.get("path").toString();
						if(filePath!=null){
							Boolean falg=deleteFile(filePath);
						}
					}
				}
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}

		}
	}

	// hql 分页查询入口
	public JSONArray hqlQuery(String name, int pageNo, int pageSize, String str)
			throws Exception {
		// pageNo小于0， 纠正成1
		if (pageNo <= 0) {
			pageNo = 1;
		}

		// pageSize小于0，纠正成1
		if (pageSize < 1 || pageSize > 1000) {
			pageSize = 1000;
		}

		// 拿到json对象参数
		JSONObject param = null;
		if (str != null && !str.isEmpty()) {
			param = new JSONObject(str);
			param = param.getJSONObject("data");
		} else {
			param = new JSONObject();
		}
		Map<String, Object> params = JsonHelper.toMap(param);
		return hqlQuery(name, pageNo, pageSize, params);
	}

	// hql 分页查询入口，取消pageSize大于1000时默认为1000的限制
	public JSONArray hqlQueryAll(String name, int pageNo, int pageSize, String str)
			throws Exception {
		// pageNo小于0， 纠正成1
		if (pageNo <= 0) {
			pageNo = 1;
		}

		// pageSize小于0，纠正成1
		if (pageSize < 1) {
			pageSize = 1000;
		}

		// 拿到json对象参数
		JSONObject param = null;
		if (str != null && !str.isEmpty()) {
			param = new JSONObject(str);
			param = param.getJSONObject("data");
		} else {
			param = new JSONObject();
		}
		Map<String, Object> params = JsonHelper.toMap(param);
		return hqlQuery(name, pageNo, pageSize, params);
	}

	// 执行分页查询
	public JSONArray hqlQuery(String name, int pageNo, int pageSize,
							  Map<String, Object> params) throws Exception {

		String sql = this.call(name, params);

		Session session = (assignedSession==null ? sessionPool.getSession() : assignedSession);
		JSONArray array = SqlHelper
				.hqlQuery(session, sql, pageNo - 1, pageSize);
		log.info("SQL["+ name+"] Result: \n"+ array.toString());
		return array;
	}

	// 执行基本hql查询
	public JSONArray hqlQuery(String name, JSONObject params) throws Exception {
		Map<String, Object> map = JsonHelper.toMap(params);
		String sql = this.call(name, map);

		Session session = (assignedSession == null ? sessionPool.getSession() : assignedSession);
		return SqlHelper.hqlQuery(session, sql);

	}

	// 直接执行基本hql查询
	public JSONArray hqlQuery(String hql) throws Exception {
		Session session = (assignedSession == null ? sessionPool.getSession() : assignedSession);
		return SqlHelper.hqlQuery(session, hql);

	}

	// 过滤order by子句，产生求和结果
	public String filterOutOrderBy(String source, Object sums)
			throws Exception {
		int idx = source.toLowerCase().lastIndexOf("order by");
		if(idx == -1) {
			idx = source.length() - 1;
		}
		StringBuilder sql = new StringBuilder("select ");
		// 如果有求和部分，产生求和部分的语句
		if (sums != null) {
			if (sums instanceof JSONArray) {
				JSONArray arraySums = (JSONArray) sums;
				for (int i = 0; i < arraySums.length(); i++) {
					String name = (String) arraySums.get(i);
					sql.append("sum(").append(name).append(") ").append(name).append(", ");
				}
			}else if (sums instanceof JSONObject) {
				JSONObject objSums = (JSONObject) sums;
				Iterator<String> keys = objSums.keys();
				while (keys.hasNext()) {
					String name = keys.next();
					String value = objSums.getString(name);
					sql.append("sum(").append(value).append(") ").append(name).append(", ");
				}
			}
		}
		if (idx != -1) {
			sql.append("count(*) n, 1 placeholder from ( ").append(source, 0, idx).append(") t_");
		}
		return sql.toString();
	}

	/**
	 * 模拟平板端执行过程
	 */
	public JSONObject action(JSONObject json) {
		JSONObject result = new JSONObject();
		try{
			if(json.has("alias")) {
				JSONObject data = new JSONObject();
				if(json.has("count")) {
					String sumJSON = this.call(json.getString("alias"), json);
					result.put("code", 200);
					data.put("sum", sumJSON);
					result.put("data", data);
				}
				JSONArray array;
				if(json.has("pageNo")) {
					Map<String, Object> params = toMap(json.getJSONObject("criteria"));
					array = this.hqlQuery(json.getString("alias"),  json.getInt("pageNo"), json.getInt("pageSize"),params);
					result.put("code", 200);
					data.put("rows", array);
					result.put("data", data);
					log.debug("得到数据" + data);
				} else {
					Map<String, Object> params = toMap(json.getJSONObject("criteria"));
					array = this.hqlQuery(json.getString("alias"), -1, -1,params);
					result.put("code", 200);
					data.put("rows", array);
					result.put("data", data);
					log.debug("得到数据" + data);
				}
			} else if(json.has("cmd")) {
				this.run(json.getString("sql"));
				result.put("code", 200);
				result.put("data", "");
			} else if(json.has("cmd2")) {
				this.run(json.getString("sql"),"del");
				result.put("code", 200);
				result.put("data", "");
			} else {
				JSONArray rows = this.querySQL(json.getString("sql"));
				result.put("code", 200);
				result.put("data", rows);
			}
		}
		catch (RuntimeException e) {
			throw e;
		} catch (Exception e) {
			throw new RuntimeException("执行错误！", e);
		}
		return result;
	}

	public void openStandardTransformerSupport() {
		SQLRuntimeSupport.openStandardTransformer();
	}

	public void closeStandardTransformerSupport() {
		SQLRuntimeSupport.closeStandardTransformer();
	}
}
