package com.aote.config;

import com.aote.exception.FileNotFoundException;
import com.aote.util.other.JsonTools;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONObject;

import java.util.Optional;

/**
 * 系统配置
 *
 * @author Mr.river
 */
public class SystemConfig {
    private static final Logger LOGGER = Logger.getLogger(SystemConfig.class);

    public static final String CONFIG_PATH = "systemConfig.json";

    private static boolean isLoaded = false;

    /**
     * 获取配置文件并加载
     */
    private static void load() {
        //获取配置文件
        if(!isLoaded){
            isLoaded = true;
            if (SystemConfig.class.getClassLoader().getResource(CONFIG_PATH) != null) {
                JSONObject params = JsonTools.readJsonFile(CONFIG_PATH);
                /* 系统配置 **/
                JSONObject system = params.getJSONObject("system");
                if(!system.has("webAppName")){
                    throw new RuntimeException(CONFIG_PATH+"中服务应用模块名参数[system:webAppName]不存在，请配置该参数为web应用名称，如revenue,webmeter等");
                }
                System.webAppName = system.getString("webAppName");
                SaleVersionEnum saleVersionEnum;
                if(!system.has("saleVersion")){
                    LOGGER.warn(CONFIG_PATH+"中服务应用模块名参数[system:saleVersion]不存在，默认为Vue-V3.");
                    saleVersionEnum = SaleVersionEnum.Vue_V3;
                } else {
                    saleVersionEnum = SaleVersionEnum.toType(system.getString("saleVersion"));
                    if(saleVersionEnum == null){
                        throw new RuntimeException(CONFIG_PATH+"中服务应用模块名参数[system:saleVersion]配置有误，不存在的值：" + system.getString("saleVersion"));
                    }
                }
                System.saleVersion = saleVersionEnum;
                if(system.has("proxyUrl")){
                    System.proxyUrl = system.getString("proxyUrl");
                }
                if(system.has("ldapUrl")){
                    System.ldapUrl = system.getString("ldapUrl");
                }
                /* 日志持久化配置 **/
                JSONObject persistenceLog = params.getJSONObject("persistenceLog");
                PersistenceLog.isEnabled = persistenceLog.optBoolean("isEnabled", false);
                if(PersistenceLog.isEnabled){
                    LOGGER.info("异常日志持久化记录已开启，如需关闭，请将"+CONFIG_PATH+"中[persistenceLog:isEnabled]配置为false.");
                } else {
                    LOGGER.warn("异常日志持久化记录已关闭，如需开启，请将"+CONFIG_PATH+"中[persistenceLog:isEnabled]配置为true.");
                }
                /* Redis配置 **/
                JSONObject redis = params.getJSONObject("redis");
                Redis.hostName = redis.optString("hostName", null);
                Redis.password = redis.optString("password", null);
                if(redis.has("codecType")){
                    Redis.codecType = redis.getString("codecType");
                }
                if (!redis.has("registerUrl")) {
                    LOGGER.warn("业务redis注册地址未配置，部分功能将无法正常使用");
                } else {
                    Redis.registerUrl = redis.getString("registerUrl");
                }
                /* 资源类配置 **/
                JSONObject resources = params.getJSONObject("resources");
                if(!resources.has("fileRootPath")){
                    LOGGER.warn(CONFIG_PATH + "中资源根目录参数[resources:fileRootPath]不存在，以按照缺省值[D:/"+ System.webAppName + "Res]配置");
                }
                Resources.fileRootPath = resources.optString("fileRootPath", "D:/" + System.webAppName + "Res");
                //  如果有扩展目录配置，则追加
                if(resources.has("extraPaths")){
                    Resources.extraPaths = resources.getJSONObject("extraPaths");
                }
                /* 实体映射配置 **/
                SubClass.subClassArray = params.optJSONArray("subclass");
                /* 数据流配置 **/
                Dataflow.isEnabled = params.optBoolean("isEnabled", false);
                Dataflow.partitionTables = params.optJSONArray("partitionTables");
                /* 数据安全配置 **/
                JSONObject dataSecurity = params.optJSONObject("dataSecurity");
                if (Optional.ofNullable(dataSecurity).isPresent()) {
                    DataSecurity.enableDataEncryption = dataSecurity.optBoolean("enableDataEncryption", false);
                    DataSecurity.aesKey = dataSecurity.optString("aesKey");
                    DataSecurity.injectionCheck = dataSecurity.optBoolean("injectionCheck", false);
                    DataSecurity.passUrl = dataSecurity.optString("passUrl");
                } else {
                    DataSecurity.enableDataEncryption = false;
                    DataSecurity.injectionCheck = false;
                }
            } else {
                throw new FileNotFoundException("系统配置文件"+CONFIG_PATH+"不存在", CONFIG_PATH);
            }
        }
    }

    /**
     * 系统
     */
    public static class System {

        private static String webAppName;

        private static SaleVersionEnum saleVersion;

        private static String proxyUrl;

        private static String ldapUrl;

        static {
            SystemConfig.load();
        }

        public static String getWebAppName() {
            return webAppName;
        }

        public static SaleVersionEnum saleVersion() {
            return saleVersion;
        }

        public static String getProxyUrl() {
            return proxyUrl;
        }

        public static boolean hasProxyUrl() {
            return proxyUrl != null;
        }

        public static String getLdapUrl() {
            return ldapUrl;
        }

        public static boolean hasLdapUrl() {
            return ldapUrl == null;
        }
    }

    /**
     * 日志持久化
     */
    public static class PersistenceLog {
        private static Boolean isEnabled;

        static {
            SystemConfig.load();
        }

        public static Boolean getEnabled() {
            return isEnabled;
        }
    }

    /**
     * Redis
     */
    public static class Redis {
        private static String hostName;
        private static String password;

        private static String registerUrl;

        private static String codecType;

        static {
            SystemConfig.load();
        }

        public static String getHostName() {
            return hostName;
        }
        public static boolean hasHostName() {
            return hostName != null;
        }
        public static String getPassword() {
            return password;
        }
        public static boolean hasPassword() {
            return password != null;
        }
        public static String getRegisterUrl() {
            return registerUrl;
        }
        public static boolean hasRegisterUrl() {
            return registerUrl != null;
        }

        public static String getCodecType() {
            return codecType;
        }

        public static boolean hasCodecType() {
            return codecType != null;
        }
    }

    /**
     * 资源
     */
    public static class Resources {
        private static String fileRootPath;
        private static JSONObject extraPaths;

        static {
            SystemConfig.load();
        }

        public static String getFileRootPath() {
            return fileRootPath;
        }

        public static String getExtraPath(String key) {
            return extraPaths.getString(key);
        }
    }

    /**
     * 实体映射
     */
    public static class SubClass {
        private static JSONArray subClassArray;

        static {
            SystemConfig.load();
        }

        public static JSONArray getSubClassArray() {
            return subClassArray;
        }
        public static boolean hasSubClassArray() {
            return subClassArray != null;
        }
    }

    public static class Dataflow {
        private static Boolean isEnabled;
        private static JSONArray partitionTables;

        static {
            SystemConfig.load();
        }

        public static Boolean getEnabled() {
            return isEnabled;
        }


        public static JSONArray getPartitionTables() {
            return partitionTables;
        }

        public static boolean hasPartitionTables(){
            return partitionTables != null;
        }
    }

    /**
     * 数据安全
     */
    public static class DataSecurity {
        /**
         * 启用数据加密
         */
        private static Boolean enableDataEncryption;
        /**
         * AES密钥
         */
        private static String aesKey;
        /**
         * SQL注入检查
         */
        private static Boolean injectionCheck;

        /**
         * 放行的url,以-分隔,例如: rs/search-rs/sql/abc
         */
        private static String passUrl;

        static {
            SystemConfig.load();
        }
        public static boolean getEnableDataEncryption() { return enableDataEncryption; }
        public static String getAesKey() { return aesKey; }
        public static boolean hasAesKey() { return aesKey != null; }
        public static boolean getInjectionCheck() { return injectionCheck; }
        public static String getPassUrl() {
            return passUrl;
        }
    }
}
