package com.aote.rs;

import com.aote.logic.LogicServer;
import com.aote.mq.RocketMQProducer;
import com.aote.util.ResourceHelper;
import com.aote.util.SnowflakeIdWorker;
import com.aote.util.other.JsonTools;
import org.apache.log4j.Logger;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.common.message.Message;
import org.apache.rocketmq.remoting.exception.RemotingException;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.inject.Singleton;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import java.nio.charset.StandardCharsets;
import java.util.*;

@Path("logic")
@Singleton
@Component
public class LogicService {
	static Logger log = Logger.getLogger(LogicService.class);
	static Logger reviewLog = Logger.getLogger("reviewLog");

	@Autowired
	private LogicServer logicServer;

	@Autowired
	private RocketMQProducer producer;

	@Autowired
	private ResourceHelper resourceHelper;

	public static final ThreadLocal<Long> reviewKey = new ThreadLocal<>();
	public static final Map<Long, JSONArray> reviewCache = new HashMap<>();

	/**
	 * 清理Logic源缓存
	 * @return ok
	 */
	@GET
	@Path("/clear")
	public String clearLogicCache(){
		ResourceHelper.clearLogicCache();
		return "ok";
	}

	/**
	 * 清理所有缓存
	 * @return ok
	 */
	@GET
	@Path("/clearall")
	public String clearAllCache(){
		ResourceHelper.clearLogicCache();
		ResourceHelper.clearSqlCache();
		ResourceHelper.clearPathCache();
		resourceHelper.clearQueryParamsCache();
		return "ok";
	}

	/**
	 * 执行业务逻辑(GET方式)
	 *
	 * @param logicName 业务逻辑名称
	 * @param request   HttpServletRequest对象
	 * @return 执行结果
	 */
	@GET
	@Path("{logic}")
	public Response xtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) {
		Map paramMap = request.getParameterMap();
		JSONObject params = new JSONObject(paramMap);
		String values = params.toString();
		return runMain(logicName, values);
	}

	/**
	 * 执行业务逻辑(POST方式)
	 *
	 * @param logicName 业务逻辑名称
	 * @param values JSON资源字符串
	 * @return 执行结果
	 */
	@POST
	@Path("{logic}")
	public Response xtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request) {
		values = getValues(values,request);
		return runMain(logicName, values);
	}

	@GET
	@Path("mq/{logic}")
	public Response mqXtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) throws Exception {
		Map paramMap = request.getParameterMap();
		JSONObject params = new JSONObject(paramMap);
		String values = params.toString();
		return runMQ(logicName, values);
	}

	@POST
	@Path("mq/{logic}")
	public Response mqXtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request) throws Exception {
		values = getValues(values,request);
		return runMQ(logicName, values);
	}

	/**
	 * 异步执行业务逻辑(GET方式)
	 * https://stackoverflow.com/questions/31401640/spring-xml-equivalent-of-enableasync
	 *
	 * @param logicName 业务逻辑名称
	 * @param request   HttpServletRequest对象
	 * @return 执行结果
	 */
	@GET
	@Path("async/{logic}")
	public Response asyncXtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) {
		Map paramMap = request.getParameterMap();
		JSONObject params = new JSONObject(paramMap);
		try {
			runAsync(logicName, params.toString());
		} catch (Exception ignore){}
		//移动平台推送特殊处理
		if(params.has("msg")){
			return Response.ok(params.getJSONArray("msg").get(0)).build();
		}
		return Response.ok().build();
	}

	/**
	 * 异步执行业务逻辑(POST方式)
	 * https://stackoverflow.com/questions/31401640/spring-xml-equivalent-of-enableasync
	 *
	 * @param logicName 业务逻辑名称
	 * @param values JSON资源字符串
	 * @return 执行结果
	 */
	@POST
	@Path("async/{logic}")
	public Response asyncXtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request) {
		try {
			runAsync(logicName, values);
		} catch (Exception ignore){}
		return Response.ok().build();
	}

	private String getValues(String values,HttpServletRequest request){
		if (!values.startsWith("{")){
			JSONObject paramJson = new JSONObject();
			paramJson.put("str",values);
			//加入请求头数据
			Enumeration headerNames = request.getHeaderNames();
			if(headerNames != null){
				JSONObject header = new JSONObject();
				while (headerNames.hasMoreElements()){
					String temp = headerNames.nextElement().toString();
					header.put(temp, request.getHeader(temp));
				}
				paramJson.put("header",header);
			}
			values = paramJson.toString();
		}
		return values;
	}

	public String xtSave(String logicName, String values) {
		return xtSave(logicName,values,null).getEntity().toString();
	}

	private Response runMain(String logicName, String values){
		long id = SnowflakeIdWorker.getValue();
		reviewKey.set(id);
		long begin = System.currentTimeMillis();
		JSONObject reviewObj = new JSONObject();
		reviewCache.put(id, new JSONArray());
		try {
			String result = logicServer.runMain(logicName, values);
			Response.ResponseBuilder builder = Response.ok();
			if (result.startsWith("{")) {
				JSONObject paramJson = new JSONObject(result);
				//如果Logic返回值中包含header和content关键字，则认为需要给调用方指定的响应头
				if (paramJson.has("header") && paramJson.has("content")) {
					Map<String, Object> paramHeader = paramJson.getJSONObject("header").toMap();

					for (Map.Entry<String, Object> stringObjectEntry : paramHeader.entrySet()) {
						builder.header(stringObjectEntry.getKey(), String.valueOf(stringObjectEntry.getValue()));
					}
					return builder.type("text/plain").entity(String.valueOf(paramJson.get("content"))).build();
				}
			}
			reviewObj.put("status", 0);
			return builder.entity(result).build();
		} catch (Exception e){
			reviewObj.put("status",-1);
			e.printStackTrace();
		} finally {
			long end = System.currentTimeMillis();
			reviewObj.put("bTime", begin);
			reviewObj.put("eTime", end);
			reviewObj.put("constTime", end - begin);
			reviewObj.put("type", "logic");
			reviewObj.put("items", reviewCache.get(id));
			reviewLog.info(reviewObj);
			reviewKey.remove();
			reviewCache.remove(id);
		}
		return Response.serverError().build();
	}

	private Response runMQ(String logicName, String values) throws RemotingException, InterruptedException, MQClientException {
		Message msg = new Message("logic",logicName,values.getBytes(StandardCharsets.UTF_8));
		// 发送单向消息
		producer.getDefaultMQProducer().sendOneway(msg);
		//移动平台推送特殊处理
		if(values.startsWith("{")){
			JSONObject params = new JSONObject(values);
			if(params.has("msg")) {
				return Response.ok(params.getJSONArray("msg").get(0)).build();
			}
		}
		return Response.ok().build();
	}

	private void runAsync(String logicName, String values){
		logicServer.runAsync(logicName, values);
	}
}
