package com.aote.redis;

import com.aote.rs.mapper.WebException;
import com.aote.util.other.JsonTools;
import org.apache.log4j.Logger;
import org.json.JSONObject;
import org.redisson.Redisson;
import org.redisson.api.RBucket;
import org.redisson.api.RLock;
import org.redisson.api.RMap;
import org.redisson.api.RedissonClient;
import org.redisson.config.Config;
import org.redisson.config.SingleServerConfig;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * Redis操作工具
 *
 * @author Mr.river
 */
@Component
public class RedisUtil implements RedisService {
    //redis操作类
    private static final RedissonClient redissonClient;
    private static final Logger log = Logger.getLogger(RedisUtil.class);

    static {
        //加载redis配置文件
        JSONObject redisConfigMap = null;
        if (RedisUtil.class.getClassLoader().getResourceAsStream("redisConfig.json") != null) {
            redisConfigMap = JsonTools.readJsonFile("redisConfig.json");
        }
        Config config = new Config();
        SingleServerConfig singleServerConfig = config.useSingleServer();
        if(redisConfigMap != null) {
            singleServerConfig.setAddress(
                    "redis://"+redisConfigMap.getString("hostName")
            );
            if (redisConfigMap.has("password")) {
                singleServerConfig.setPassword(redisConfigMap.getString("password"));
            }
            redissonClient = Redisson.create(config);
            log.info("***redis实例已加载完成***");
        } else {
            redissonClient = null;
            log.warn("***redis实例未配置***");
        }
    }

    @Override
    public void set(String key, Object value) {
        set(key,value,-1);
    }

    @Override
    public void set(String key, Object value, long timeoutSec) {
        RBucket<String> rBucket = redissonClient.getBucket(key);
        if(timeoutSec > 0){
            rBucket.set(String.valueOf(value),timeoutSec, TimeUnit.SECONDS);
        } else {
            rBucket.set(String.valueOf(value));
        }
    }

    @Override
    public void setHash(String hashKey, JSONObject value) {
        setHash(hashKey, value.toMap());
    }

    @Override
    public void setHash(String hashKey, Map<String, Object> value) {
        setHash(hashKey,value,-1);
    }

    @Override
    public void setHash(String hashKey, Map<String, Object> value, long timeoutSec) {
        RMap<String, Object> rMap = redissonClient.getMap(hashKey);
        rMap.putAll(value);
        if(timeoutSec > 0){
            rMap.expire(timeoutSec, TimeUnit.SECONDS);
        }
    }

    @Override
    public JSONObject getHash(String hashKey) {
        return new JSONObject(redissonClient.getMap(hashKey));
    }

    @Override
    public Object getHash(String hashKey, String key) {
        return redissonClient.getMap(hashKey).get(key);
    }

    @Override
    public void setHashKey(String hashKey, String key, Object value) {
        redissonClient.getMap(hashKey).put(key,value);
    }

    @Override
    public void deleteHashKey(String hashKey, String key) {
        redissonClient.getMap(hashKey).remove(key);
    }

    @Override
    public Boolean hasHashKey(String hashKey, String key) {
        return redissonClient.getMap(hashKey).containsKey(key);
    }

    @Override
    public void lock(String key, Runnable runnable) {
        RLock lock = redissonClient.getLock("LOCK-"+key);
        boolean isLock;
        try{
            isLock = lock.tryLock(10,20,TimeUnit.SECONDS);
            if(isLock){
                runnable.success();
            } else {
                throw new WebException(505, "修改失败，该数据受锁保护中。");
            }
        } catch (InterruptedException e) {
            e.printStackTrace();
        } finally {
            lock.unlock();
        }
    }

    @Override
    public Object get(String key) {
        return redissonClient.getBucket(key).get();
    }

    @Override
    public void delete(String key) {
        redissonClient.getBucket(key).delete();
    }

    @Override
    public void deleteList(Iterable<Object> keys) {
        keys.forEach((item) -> delete(String.valueOf(item)));
    }

    @Override
    public void deleteList(JSONObject keys) {
        deleteList(new ArrayList<>(keys.keySet()));
    }

    @FunctionalInterface
    public interface Runnable {
        void success();
    }
}
