package com.aote.logic;

import com.aote.entity.EntityServer;
import com.aote.exception.FileNotFoundException;
import com.aote.redis.RedisUtil;
import com.aote.sql.SqlMapper;
import com.aote.sql.SqlServer;
import com.aote.transaction.SessionPool;
import com.aote.util.ExpressionHelper;
import com.aote.util.ResourceHelper;
import com.aote.util.ResourceType;
import org.apache.log4j.Logger;
import org.hibernate.SessionFactory;
import org.hibernate.metadata.ClassMetadata;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import redis.clients.jedis.exceptions.JedisConnectionException;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

@Component
@Transactional
public class LogicServer {
	static Logger log = Logger.getLogger(LogicServer.class);

	@Autowired
	private EntityServer entityServer;

	@Autowired
	private SqlServer sqlServer;

	@Autowired
	private RedisUtil redisUtil;

	@Autowired
	public SessionPool sessionPool;

	@Autowired
	public SessionFactory sessionFactory;

	// 安卓模拟时需要的上下文
	private JSONObject context = null;

	// 执行业务逻辑处理过程
	public Object run(String name, String str) {
		JSONObject param = null;
		// 把传递过来的参数，放到data里，以便跟entity，sql等对象区别开来
		// 判断传来的数据是否是XML格式
		str = str.trim();
		if (str.startsWith("<") && str.endsWith(">")) {
			// 传来数据为XML格式
			param = new JSONObject();
			param.put("xml", str);
		} else {
			// 传来数据为JSON格式
			Object json = new JSONTokener(str).nextValue();
			if (json instanceof JSONObject) {
				param = new JSONObject(str);
				// 有context部分，取context部分
				if (!param.isNull("context")) {
					context = param.getJSONObject("context");
				}
				if (!param.isNull("data")) {
					Object dataParam = param.get("data");
					if (dataParam instanceof JSONObject){
						param = (JSONObject) dataParam;
						param.put("standardData",new JSONObject(str).toString());
					}
				}
			} else if (json instanceof JSONArray) {
				param = new JSONObject();
				param.put("arr", new JSONArray(str));
			}
		}

		return run(name, param, context);
	}

	// 业务逻辑带一个上下问对象，用于平板端测试
	public Object run(String name, JSONObject param, JSONObject context) {
		// 获取源程序内容
		String path = LogicMapper.getLogic(name);
		if (path == null) {
			throw new RuntimeException("业务逻辑未找到: " + name);
		}

		String source;
		try {
			source = ResourceHelper.getString(ResourceType.LOGIC,name,path);
		} catch (FileNotFoundException e) {
			throw new RuntimeException(path + ".文件无配置");
		} catch (IOException io) {
			throw new RuntimeException(io);
		}
		// 执行源程序
		Map<String, Object> params = new HashMap<>();
		params.put("data", param);
		// context用于测试平板端业务逻辑
		if (context != null) {
			params.put("context", context);
		}
		// 附加entityServer, sqlServer等对象到参数中
		params.put("log", log);
		params.put("entity", entityServer);
		params.put("sql", sqlServer);
		params.put("session", sessionPool.getSession());
		params.put("redis",redisUtil);
		// 把LogicServer自身注册进去
		params.put("logic", this);

		// 附加用户注册的对象到业务逻辑中
		Map<String, Object> plugins = PluginMapper.getPlugins();
		params.putAll(plugins);
		return ExpressionHelper.run(source, params);
	}

	// 业务逻辑内部调用其它业务逻辑
	public Object run(String name, JSONObject param) {
		return run(name, param, this.context);
	}

	/**
	 * 将来可以根据业务逻辑名称配置
	 */
	public void debug(String logic, Object msg) {
		// 获取业务逻辑配置信息
		String level = LogicMapper.getAttr(logic, "log");
		if (level != null) {
			log.debug(logic + ":" + msg);
		}
	}
	/**
	 * redis缓存业务逻辑组件
	 */
	public void redisCacheLogic() throws Exception{
		if (ResourceHelper.class.getResourceAsStream("/config.json") == null) {
			log.info("无config.json文件，不进行redis缓存");
			return;
		}
		String strConfig = ResourceHelper.getString("config.json");
		JSONObject jsonObject = new JSONObject(strConfig);
		if (!jsonObject.has("logicRegister")) {
			log.info("config.json中无模块镜像，不进行redis缓存");
			return;
		}
		//获取模块镜像IP和端口号
		JSONObject logicJsonObject = jsonObject.getJSONObject("logicRegister");
		String url = logicJsonObject.getString("url");
		if (url == null){
			log.info("config.json中无url，不进行redis缓存");
			return;
		}

		try {
			//获取模块逻辑名称
			Map<String, Map<String, String>> map = LogicMapper.getMap();
			if (map != null) {
				//遍历逻辑集合，将逻辑及对应IP和端口号存入服务器的redis缓存中
				for (String key : map.keySet()) {
					redisUtil.set(key, url);
					log.info("logic加入redis缓存，key=" + key + ", value=" + url);
				}
			}
			// 对sql语句进行注册
			map = SqlMapper.getMap();
			if (map != null) {
				//遍历逻辑集合，将逻辑及对应IP和端口号存入服务器的redis缓存中
				for (String key : map.keySet()) {
					redisUtil.set(key, url);
					log.info("sql加入redis缓存，key=" + key + ", value=" + url);
				}
			}
			// 对实体进行注册
			Map<String, ClassMetadata> entities = sessionFactory.getAllClassMetadata();
			for (String key : entities.keySet()) {
				redisUtil.set(key, url);
				log.info("entity加入redis缓存，key=" + key + ", value=" + url);
			}
		} catch(JedisConnectionException e) {
			log.warn("无法连接redis服务器，未成功注册业务逻辑等组件！");
		}
	}
}
