package com.aote.rs;

import com.aote.exception.ExceptionLogManage;
import com.aote.logic.LogicMapper;
import com.aote.logic.LogicServer;
import com.aote.mq.RocketMQProducer;
import com.aote.rs.mapper.WebException;
import com.aote.transaction.SessionPool;
import com.aote.util.ExceptionHelper;
import com.aote.util.ResourceHelper;
import org.apache.log4j.Logger;
import org.apache.rocketmq.common.message.Message;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import javax.annotation.PostConstruct;
import javax.inject.Singleton;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import java.nio.charset.StandardCharsets;
import java.util.Enumeration;
import java.util.Map;

@Path("logic")
@Singleton
@Component
public class LogicService {
	static Logger log = Logger.getLogger(LogicService.class);

	@Autowired
	private SessionPool sessionPool;

	@Autowired
	private LogicServer logicServer;

	@Autowired
	@Qualifier("rocketMQProducer")
	private RocketMQProducer producer;

	@Autowired
	private ExceptionLogManage exceptionLogManage;

	/**
	 * 清理Logic源缓存
	 * @return ok
	 */
	@GET
	@Path("/clear")
	public String clearLogicCache(){
		ResourceHelper.clearLogicCache();
		ResourceHelper.clearSqlCache();
		ResourceHelper.clearPathCache();
		return "ok";
	}

	/**
	 * 执行业务逻辑(GET方式)
	 *
	 * @param logicName 业务逻辑名称
	 * @param request   HttpServletRequest对象
	 * @return 执行结果
	 */
	@GET
	@Transactional
	@Path("{logic}")
	public String xtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) throws Exception {
		Map paramMap = request.getParameterMap();
		JSONObject params = new JSONObject(paramMap);
		return xtSave(logicName, params.toString(),request);
	}

	/**
	 * 执行业务逻辑(POST方式)
	 *
	 * @param logicName 业务逻辑名称
	 * @param values JSON资源字符串
	 * @return 执行结果
	 */
	@POST
	@Transactional
	@Path("{logic}")
	public String xtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request) {
		log.info("logic:" + logicName + ", values:" + values);
		values = getValues(values,request);
		return run(logicName, values);
	}

	@GET
	@Transactional
	@Path("mq/{logic}")
	public Response mqXtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) throws Exception {
		Map paramMap = request.getParameterMap();
		JSONObject params = new JSONObject(paramMap);
		return mqXtSave(logicName, params.toString(),request);
	}

	@POST
	@Transactional
	@Path("mq/{logic}")
	public Response mqXtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request)
			throws Exception {
		log.info("logic:" + logicName + ", values:" + values);
		values = getValues(values,request);
		Message msg = new Message("logic",logicName,values.getBytes(StandardCharsets.UTF_8));
		// 发送单向消息
		producer.getDefaultMQProducer().sendOneway(msg);
		return Response.ok().build();
	}

	/**
	 * 异步执行业务逻辑(GET方式)
	 * https://stackoverflow.com/questions/31401640/spring-xml-equivalent-of-enableasync
	 *
	 * @param logicName 业务逻辑名称
	 * @param request   HttpServletRequest对象
	 * @return 执行结果
	 */
	@GET
	@Transactional
	@Async("AsyncLogicGet")
	@Path("async/{logic}")
	public String asyncXtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) {
		Map paramMap = request.getParameterMap();
		JSONObject params = new JSONObject(paramMap);
		return xtSave(logicName, params.toString(),request);
	}

	/**
	 * 异步执行业务逻辑(POST方式)
	 * https://stackoverflow.com/questions/31401640/spring-xml-equivalent-of-enableasync
	 *
	 * @param logicName 业务逻辑名称
	 * @param values JSON资源字符串
	 * @return 执行结果
	 */
	@POST
	@Async("AsyncLogicPost")
	@Transactional
	@Path("async/{logic}")
	public void asyncXtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request)
			throws Exception {
		xtSave(logicName,values,request);
	}

	/**
	 * 执行调试用的业务逻辑(GET方式)，自动回滚
	 *
	 * @param logicName 业务逻辑名称
	 * @param request   HttpServletRequest对象
	 * @return 执行结果
	 */
	@GET
	@Transactional
	@Path("debug/{logic}")
	public String debugXtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) throws Exception {
		Map paramMap = request.getParameterMap();
		JSONObject params = new JSONObject(paramMap);
		return debugXtSave(logicName, params.toString(),request);
	}

	/**
	 * 执行调试用的业务逻辑(POST方式)，自动回滚
	 *
	 * @param logicName 业务逻辑名称
	 * @param values JSON资源字符串
	 * @return 执行结果
	 */
	@POST
	@Transactional
	@Path("debug/{logic}")
	public String debugXtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request)
			throws Exception {
		String result = xtSave(logicName,values,request);
		//事务回滚
		TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
		return result;
	}

	@Transactional
	public String xtSave(String logicName, String values) throws Exception {
		return xtSave(logicName,values,null);
	}

	private String getValues(String values,HttpServletRequest request){
		if (!values.startsWith("{")){
			JSONObject paramJson = new JSONObject();
			paramJson.put("str",values);
			//加入请求头数据
			Enumeration headerNames = request.getHeaderNames();
			if(headerNames != null){
				JSONObject header = new JSONObject();
				while (headerNames.hasMoreElements()){
					String temp = headerNames.nextElement().toString();
					header.put(temp, request.getHeader(temp));
				}
				paramJson.put("header",header);
			}
			values = paramJson.toString();
		}
		return values;
	}

	/**
	 * 运行业务逻辑
	 * @param logicName 业务逻辑名
	 * @param values 业务逻辑参数
	 * @return 运行结果
	 */
	private String run(String logicName, String values) {
		try {
			Object result = logicServer.run(logicName, values);
			if (result == null) {
                return "";
			}
			// 如果执行结果为Map，转换成JSON串
			if (result instanceof Map<?, ?>) {
				JSONObject json = new JSONObject((Map<?, ?>) result);
				return json.toString();
			}
			if (result instanceof JSONObject) {
				JSONObject json = (JSONObject) result;
				return json.toString();
			}
			return result.toString();
		} catch (Exception ex) {
			String stack = ExceptionLogManage.getErrorInfo(ExceptionHelper.stackToString(ex),1);
			log.error(stack);
			// 获取源程序路径
			String path = LogicMapper.getLogic(logicName);
			if(path == null){
				path = logicName;
			}
			exceptionLogManage.writeError(ex,path,"/rs/logic/"+logicName);
			WebException wex = this.getWebException(ex);
			if (wex != null)
			{
				throw wex;
			}
			throw ex;
		}
	}

	// 找到WebException异常
	// return - 不存在，返回null
	private WebException getWebException(Exception ex)
	{
		Throwable cause = ex;
		while (cause != null)
		{
			if (cause instanceof WebException)
			{
				return (WebException)cause;
			}
			cause = cause.getCause();
		}

		return null;
	}

	//服务器启动时加载模块至redis
	@PostConstruct
	public void redisCacheLogic() throws Exception {
		logicServer.redisCacheLogic();
	}
}
