package com.aote.queryparams;

import com.aote.exception.FileNotFoundException;
import com.aote.rs.mapper.WebException;
import com.aote.util.ResourceHelper;
import com.aote.util.ResourceType;
import org.apache.commons.lang3.StringUtils;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.util.Iterator;

@Component
public class QueryParamsServer {
    public static JSONObject getQuerySQL(String queryParamsName, JSONObject params) throws Exception {
        String path = QueryParamsMapper.getQueryParams(queryParamsName);
        if (path == null) {
            throw new RuntimeException("查询参数文件未找到: " + queryParamsName);
        }
        JSONObject queryParamsMap;
        try {
            queryParamsMap = (JSONObject) ResourceHelper.getString(ResourceType.QUERY_PARAMS,queryParamsName,path);
        } catch (FileNotFoundException e) {
            throw new RuntimeException(path + ".文件无配置");
        } catch (IOException io) {
            throw new RuntimeException(io);
        }
        String tableAliasName = queryParamsMap.getString("tableAliasName");
        StringBuilder querySql = new StringBuilder(queryParamsMap.getString("querySql"));
        StringBuilder countSql = new StringBuilder(queryParamsMap.getString("countSql"));
        JSONObject joinTableNameObject = queryParamsMap.getJSONObject("joinTableNameObject");
        JSONObject canJoinTableNameObject = queryParamsMap.getJSONObject("canJoinTableNameObject");
        String orderBy = queryParamsMap.getString("orderBy");
        // 根据前台表单和查询列配置，生成where表达式
        String whereCondition = getValue(queryParamsMap.getJSONObject("selectColumn"),
                params,tableAliasName,joinTableNameObject,canJoinTableNameObject);
        // 取出所有需要JOIN的表
        Iterator<String> canJoinTableIterator = canJoinTableNameObject.keys();
        while (canJoinTableIterator.hasNext()){
            JSONObject joinObject = canJoinTableNameObject.getJSONObject(canJoinTableIterator.next());
            String condition = joinObject.getString("value");
            // 追加JOIN表和on表达式到SQL中
            querySql.append("\n\t").append(condition);
            // 根据LEFT JOIN关联方式，在count统计时,展示列用的子表不需要join，查询条件中用到的子表才需要join
            String source = joinObject.getString("source");
            if (source.equals("查询条件") || source.equals("展示列/查询条件")){
                countSql.append("\n\t").append(condition);
            }
        }
        // 追加where表达式到SQL中
        querySql.append("\n").append("WHERE").append(whereCondition);
        countSql.append("\n").append("WHERE").append(whereCondition);
        // 追加orderBy表达式到SQL中
        querySql.append("\nORDER BY ").append(orderBy);
        JSONObject result = new JSONObject();
        result.put("querySql",querySql.toString());
        result.put("countSql",countSql.toString());
        return result;
    }

    public static JSONObject getSignleStyleQuerySQL(String queryParamsName, JSONObject params) throws Exception {
        JSONObject queryParams = getQuerySQL(queryParamsName, params);
        String querySql = queryParams.getString("querySql");
        String countSql = queryParams.getString("countSql");
        JSONObject result = new JSONObject();
        result.put("items",querySql.substring(querySql.indexOf("SELECT") + 6,querySql.indexOf("FROM")));
        result.put("tableName",querySql.substring(querySql.indexOf("FROM") + 4,querySql.indexOf("WHERE")));
        result.put("condition",querySql.substring(querySql.indexOf("WHERE") + 5));
        result.put("countItems",countSql.substring(countSql.indexOf("SELECT") + 6,countSql.indexOf("FROM")));
        result.put("countTableName",countSql.substring(countSql.indexOf("FROM") + 4,countSql.indexOf("WHERE")));
        result.put("countCondition",countSql.substring(countSql.indexOf("WHERE") + 5));
        result.put("querySql", querySql);
        result.put("countSql", countSql);
        return result;
    }

    /**
     * 生成查询条件
     * @param queryMap 查询列配置JSON
     * @param params 前端传入的表单参数
     * @param tableAliasName 主表别名
     * @param joinTableNameObject 预设的关联表集合
     * @param canJoinTableNameObject 需要关联的表集合
     * @return 文件内容
     */
    public static String getValue(JSONObject queryMap, JSONObject params, String tableAliasName, JSONObject joinTableNameObject, JSONObject canJoinTableNameObject) throws Exception{
        // 循环前台传过来的条件参数
        Iterator<String> iterator = params.keys();
        // 查询条件
        StringBuilder condition = new StringBuilder(" 1=1");
        while(iterator.hasNext()) {
            String key = iterator.next();
            String value = String.valueOf(params.get(key));
            // 如果前台的条件参数有值
            if (value.length() != 0 && (!value.equals("全部"))) {
                if (queryMap.has(key)) {
                    JSONObject queryParams = queryMap.getJSONObject(key);
                    if(!queryParams.has("queryType")){
                        throw new RuntimeException("查询字段[" + key + "] 没有对应的queryType信息");
                    }
                    // 获取查询类型
                    QueryTypeEnum queryType = QueryTypeEnum.toType(queryParams.getString("queryType"));
                    // 获取该列的数据字段
                    String realKey = queryParams.getString("key");
                    switch (queryType) {
                        case EQUALS:
                        case NO_EQUALS:
                        case LESS_THAN:
                        case LESS_THAN_EQUALS:
                        case GREATER_THAN:
                        case GREATER_THAN_EQUALS:
                            condition.append("\n\tAND ").append(realKey).append(" ").append(queryType.getValue()).append(" '").append(value).append("'");
                            break;
                        case INNER_LIKE:
                            condition.append("\n\tAND ").append(realKey).append(" LIKE '%").append(value).append("%'");
                            break;
                        case LEFT_LIKE:
                            condition.append("\n\tAND ").append(realKey).append(" LIKE '%").append(value).append("'");
                            break;
                        case RIGHT_LIKE:
                            condition.append("\n\tAND ").append(realKey).append(" LIKE '").append(value).append("%'");
                            break;
                        case IN:
                            condition.append("\n\tAND ").append(realKey).append(" IN (").append(value).append(")");
                            break;
                        case NOT_IN:
                            condition.append("\n\tAND ").append(realKey).append(" NOT IN (").append(value).append(")");
                            break;
                        case BETWEEN:
                            //将参数按照,分割
                            String[] values = StringUtils.split(value.substring(1, value.length() - 1),",");
                            condition.append("\n\tAND ").append(realKey).append(" BETWEEN '").append(values[0]).append("' AND '").append(values[1]).append("'");
                            break;
                    }
                    // 追加JOIN表达式
                    putJoinCondition(realKey, tableAliasName, joinTableNameObject, canJoinTableNameObject,"查询条件");
                } else {
                    throw new WebException( 500,"生成查询条件时，缺少表单字段'"+key+"'的配置");
                }
            }
        }
        return condition.toString();
    }

    /**
     * 根据查询配置生成SQL片段以及表格展示列数据
     * @param source 查询配置源
     * @return tableAliasName 主表别名
     *         countSql countSQL片段
     *         querySql 查询SQL片段
     *         joinTableNameObject 预设的关联表集合
     *         canJoinTableNameObject 需要关联的表集合
     *         selectParamsArray [生成查询条件]所用的数据列集合
     *         columnJson 表格展示列JSON
     */
    public static JSONObject getQueryParamsInfo(String source) {
        JSONObject realQueryParamsMap = new JSONObject(source);
        // 数据列集合
        JSONArray queryParamsArray = realQueryParamsMap.getJSONArray("column");
        // 组织[生成查询条件]所用的数据列集合
        JSONObject selectColumnArray = new JSONObject();
        for(Object item : queryParamsArray){
            JSONObject columnItem = ((JSONObject) item);
            String key = columnItem.getString("key");
            int index = key.indexOf('.');
            if(index == -1){
                index = 0;
            } else {
                index = index + 1;
            }
            String realKey = key.substring(index);
            columnItem.put("dataIndex",realKey);
            selectColumnArray.put(realKey, columnItem);
        }
        realQueryParamsMap.put("column",queryParamsArray);
        // 组织查询SQL
        StringBuilder sql = new StringBuilder("SELECT ");
        // 组织查询总数SQL
        StringBuilder countSql = new StringBuilder("SELECT COUNT(0) n");
        // 查询的主表
        String tableName = realQueryParamsMap.getString("tableName");
        // 查询的主表别名
        String tableAliasName = tableName.substring(tableName.indexOf(' ') + 1);
        // 预设的关联表集合
        JSONObject joinTableNameObject = realQueryParamsMap.getJSONObject("joinArray");
        // 需要关联的表集合
        JSONObject canJoinTableNameObject = new JSONObject();
        // 查询列集合
        JSONArray queryColumnArray = realQueryParamsMap.getJSONArray("column");
        // 组织表格展示列JSON
        JSONArray showColumnArray = new JSONArray(queryColumnArray.length());
        for (Object columnItem : queryColumnArray) {
            JSONObject column = (JSONObject) columnItem;
            // 获取该列的数据字段
            String key = column.getString("key");
            // 如果是需要展示的列(has "title")
            if (column.has("title")){
                // 组织表格展示列
                JSONObject item = new JSONObject();
                item.put("title", column.getString("title"));
                item.put("dataIndex",column.getString("dataIndex"));
                if(column.has("width")){
                    item.put("width",column.getString("width"));
                }
                if(column.has("scopedSlots")){
                    item.put("scopedSlots",column.getJSONObject("scopedSlots"));
                }
                showColumnArray.put(item);
                // 如果不是"操作"列
                if (!key.equals("action")) {
                    // 追加该列的数据字段到查询SQL中
                    sql.append("\n\t").append(key).append(",");
                    // 追加JOIN表达式
                    putJoinCondition(key, tableAliasName, joinTableNameObject, canJoinTableNameObject,"展示列");
                }
            }
        }
        sql.deleteCharAt(sql.length() - 1);
        // 追加Form主表到SQL中
        sql.append("\n").append("FROM ").append(tableName);
        countSql.append("\n").append("FROM ").append(tableName);
        JSONObject result = new JSONObject();
        result.put("tableAliasName", tableAliasName);
        result.put("countSql",countSql.toString());
        result.put("querySql",sql.toString());
        result.put("joinTableNameObject",joinTableNameObject);
        result.put("canJoinTableNameObject",canJoinTableNameObject);
        result.put("selectColumn",selectColumnArray);
        result.put("orderBy",realQueryParamsMap.getString("orderBy"));
        result.put("columnJson",showColumnArray);
        return result;
    }

    /**
     * 追加JOIN表达式
     * @param dataKey 数据列
     * @param tableAliasName 主表别名
     * @param joinTableNameObject 预设的关联表集合
     * @param canJoinTableNameObject 需要关联的表集合
     * @param source 来源类型 展示列/查询条件
     */
    public static void putJoinCondition(String dataKey, String tableAliasName, JSONObject joinTableNameObject, JSONObject canJoinTableNameObject, String source){
        // 获取数据字段对应的表别名
        String joinTableAlias = dataKey.substring(0, dataKey.indexOf('.'));
        // 如果数据字段是主表的，直接忽略
        if (joinTableAlias.equals(tableAliasName)) {
            return;
        }
        // 如果数据字段对应的表在预设的关联表集合中
        if (joinTableNameObject.has(joinTableAlias)) {
            // 如果该表不在需要关联的表集合中，追加数据
            if (!canJoinTableNameObject.has(joinTableAlias)) {
                // 取到该表对应的JOIN表达式，加入到需要关联的表集合
                String joinCondition = joinTableNameObject.getString(joinTableAlias);
                JSONObject params = new JSONObject();
                params.put("source", source);
                params.put("value", "LEFT JOIN " + joinCondition);
                canJoinTableNameObject.put(joinTableAlias,params);
            } else if (source.equals("查询条件")){
                // 修改JOIN表达式的来源类型
                canJoinTableNameObject.getJSONObject(joinTableAlias).put("source","展示列/查询条件");
            }
        } else {
            throw new RuntimeException("数据列[" + dataKey + "] 没有对应的JOIN表信息");
        }
    }
}
