package com.aote.exception;

import com.aote.entity.EntityServer;
import com.aote.sql.SqlServer;
import com.aote.util.ExceptionHelper;
import com.aote.util.other.JsonTools;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * 异常日志记录管理
 */
@Component
@Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRES_NEW)
public class ExceptionLogManage {
    private static Logger LOGGER = Logger.getLogger(ExceptionLogManage.class);

    @Autowired
    private EntityServer entityServer;

    @Autowired
    private SqlServer sqlServer;

    private static final String CONFIG_PATH = "logConfig.json";

    private static Boolean isEnabled = false;

    static {
        // 获取配置文件并加载
        load();
    }

    /**
     * 获取配置文件并加载
     */
    public static void load() {
        //获取配置文件
        if (ExceptionLogManage.class.getClassLoader().getResourceAsStream(CONFIG_PATH) != null) {
            JSONObject params = JsonTools.readJsonFile(CONFIG_PATH);
            if(params.has("isEnabled") && params.getBoolean("isEnabled")){
                LOGGER.info("异常日志持久化记录已开启，如需关闭，请将logConfig.json中isEnabled参数配置为false.");
                isEnabled = true;
            } else {
                LOGGER.warn("异常日志持久化记录已关闭，如需开启，请将logConfig.json中isEnabled参数配置为true.");
                isEnabled = false;
            }
        } else {
            LOGGER.warn("异常日志持久化记录已关闭，如需开启，请在module.xml同级目录创建logConfig.json文件，并配置isEnabled参数为true.");
            isEnabled = false;
        }
    }

    public void writeError(Exception e, String path, String webUrl) {
        if(!isEnabled){
            return;
        }
        HttpServletRequest request = ((ServletRequestAttributes)
                RequestContextHolder.getRequestAttributes()).getRequest();
        String serviceName = request.getContextPath();
        String errorInfo = ExceptionHelper.stackToString(e);
        errorInfo = errorInfo.replace("'", "''");
        int causedByIndex = errorInfo.lastIndexOf("Caused by");
        String title;
        if(causedByIndex != -1){
            title = errorInfo.substring(causedByIndex);
            title = title.substring(0,title.indexOf("at "));
        } else {
            title = e.getClass() + ": "+ e.getMessage();
            title = title.replace("'", "''");
        }
        //查询是否已经记录过日志
        JSONArray logs;
        try {
            if(path == null){
                logs = sqlServer.querySQL("SELECT id,f_times FROM t_log WHERE f_service = '"+ serviceName +"' " +
                        "and f_error_title = '" + title + "'");
            } else {
                logs = sqlServer.querySQL("SELECT id,f_times FROM t_log WHERE f_service = '"+ serviceName +"' " +
                        "and f_src = '"+ path +"' and f_error_title = '" + title + "'");
            }
        } catch (Exception exception) {
            LOGGER.error(exception);
            logs = new JSONArray();
        }
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        String date = format.format(new Date());
        JSONObject params;
        if(logs.length() > 0){
            params = logs.getJSONObject(0);
            int times = params.getInt("f_times");
            int id = params.getInt("id");
            params.put("f_times",times + 1);
            params.put("f_last_update_time",date);
            params.put("id",id);
        } else {
            //精简报错堆栈内容
            String errorInfoData = getLogicErrorInfo(errorInfo);
            if(errorInfoData.equals("==================\n")){
                errorInfoData = errorInfo;
            }
            params = new JSONObject();
            params.put("f_service",serviceName);
            params.put("f_web_url",webUrl);
            params.put("f_src",path);
            params.put("f_log_type","异常");
            params.put("f_error_title",title);
            params.put("f_error_msg",errorInfoData);
            params.put("f_first_update_time",date);
            params.put("f_last_update_time",date);
            params.put("f_times",1);
        }
        try {
            entityServer.partialSave("t_log",params);
        } catch (Exception exception) {
            LOGGER.error(exception);
        }
    }

    public static String getLogicErrorInfo(String errorInfo){
        //Logic报错堆栈截取思路如下：
        //将堆栈分成两部分
        //主要部分：Logic异常报错时，会出现com.af.expression.ExpressionException异常，通过该关键词递归截取
        //结尾部分：截取Caused by，获取根堆栈信息
        String result;
        try {
            String defaultHeader = "==================\n";
            StringBuilder builder = new StringBuilder(defaultHeader);
            appendLogicErrorInfo(builder, errorInfo);
            result = builder.toString();
            if (result.equals(defaultHeader)) {
                result = errorInfo;
            }
        } catch (Exception e){
            LOGGER.error("***Logic报错堆栈截取报错***");
            result = errorInfo;
        }
        return result;
    }

    private static void appendLogicErrorInfo(StringBuilder result,String errorInfo){
        //堆栈内容
        int ignoreIndex = errorInfo.indexOf("com.af.expression.ExpressionException: ");
        if(ignoreIndex != -1){
            errorInfo = errorInfo.substring(ignoreIndex + 39);
            int ignoreIndexEnd = errorInfo.indexOf("at ");
            if(ignoreIndexEnd != -1){
                result.append(errorInfo, 0, ignoreIndexEnd).append("\n==================\n");
                errorInfo = errorInfo.substring(ignoreIndexEnd + 3);
                appendLogicErrorInfo(result, errorInfo);
            }
        } else {
            //截取结尾部分
            ignoreIndex = errorInfo.lastIndexOf("Caused by:");
            if(ignoreIndex != -1){
                result.append(errorInfo.substring(ignoreIndex));
            }
        }
    }
}
