package com.aote.exception;

import com.aote.config.SystemConfig;
import com.aote.entity.EntityServer;
import com.aote.sql.SqlServer;
import com.aote.util.ExceptionHelper;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * 异常日志记录管理
 */
@Component
@Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRES_NEW)
public class ExceptionLogManage {
    private static final Logger LOGGER = Logger.getLogger(ExceptionLogManage.class);

    @Autowired
    private EntityServer entityServer;

    @Autowired
    private SqlServer sqlServer;

    public void writeError(Exception e, String path, String webUrl, String values) {
        if(!SystemConfig.PersistenceLog.getEnabled()){
            return;
        }
        String serviceName = SystemConfig.System.getWebAppName();
        String errorInfo = ExceptionHelper.stackToString(e);
        errorInfo = errorInfo.replace("'", "''");
        int causedByIndex = errorInfo.lastIndexOf("Caused by");
        String title;
        if(causedByIndex != -1){
            title = errorInfo.substring(causedByIndex);
            title = title.substring(0,title.indexOf("at "));
        } else {
            title = e.getClass() + ": "+ e.getMessage();
            title = title.replace("'", "''");
        }
        //查询是否已经记录过日志
        JSONArray logs;
        try {
            if(path == null){
                logs = sqlServer.querySQL("SELECT id,f_times FROM t_log WHERE f_service = '"+ serviceName +"' " +
                        "and f_error_title = '" + title + "' and f_status != '已完成'",true);
            } else {
                logs = sqlServer.querySQL("SELECT id,f_times FROM t_log WHERE f_service = '"+ serviceName +"' " +
                        "and f_src = '"+ path +"' and f_error_title = '" + title + "' and f_status != '已完成'",true);
            }
        } catch (Exception exception) {
            LOGGER.error("存储异常日志报错：",exception);
            logs = new JSONArray();
        }
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        String date = format.format(new Date());
        JSONObject params;
        if(logs.length() > 0){
            params = logs.getJSONObject(0);
            int times = params.getInt("f_times");
            int id = params.getInt("id");
            params.put("f_times",times + 1);
            params.put("f_last_update_time",date);
            params.put("id",id);
        } else {
            //精简报错堆栈内容
            int type;
            if(webUrl.contains("rs/sql")){
                type = 2;
            } else {
                type = 1;
            }
            String errorInfoData = getErrorInfo(errorInfo,type);
            int maxLength;
            if(EntityServer.findDialect().equals(EntityServer.DIALECT_ORACLE)){
                maxLength = 4000;
            } else if(EntityServer.findDialect().equals(EntityServer.DIALECT_SQLSERVER)){
                maxLength = 8000;
            } else {
                maxLength = 99999;
            }
            while (errorInfoData.getBytes().length > maxLength){
                int length = errorInfoData.length();
                errorInfoData = errorInfoData.substring(0,length-60);
                errorInfoData = errorInfoData + "\n超过容量,省略后续字符...";
            }
            params = new JSONObject();
            params.put("f_service",serviceName);
            params.put("f_web_url",webUrl);
            params.put("f_params",values);
            params.put("f_src",path);
            params.put("f_log_type","异常");
            params.put("f_error_title",title);
            params.put("f_error_msg",errorInfoData);
            params.put("f_first_update_time",date);
            params.put("f_last_update_time",date);
            params.put("f_status","待办的");
            params.put("f_times",1);
        }
        try {
            entityServer.partialSave("t_log",params,true);
        } catch (Exception exception) {
            LOGGER.error(params);
            LOGGER.error("存储异常日志报错：",exception);
        }
    }

    public static String getErrorInfo(String errorInfo,int type){
        String result;
        try {
            String defaultHeader = "ERROR!======\n";
            String defaultHeaderEnd = "\nERROR_END!=====";
            StringBuilder builder = new StringBuilder(defaultHeader);
            if(type == 1) {
                //Logic报错堆栈截取思路如下：
                //将堆栈分成两部分
                //主要部分：Logic异常报错时，会出现com.af.expression.ExpressionException异常，通过该关键词递归截取
                //结尾部分：截取Caused by，获取根堆栈信息
                appendLogicErrorInfo(builder, errorInfo);
            } else {
                //Sql报错堆栈截取思路如下：
                //将堆栈分成两部分
                //主要部分：SQL异常报错时，会出现org.hibernate.exception.SQLGrammarException异常，通过该关键词递归截取
                //结尾部分：截取Caused by，获取根堆栈信息
                appendSqlErrorInfo(builder, errorInfo);
            }
            builder.append(defaultHeaderEnd);
            result = builder.toString();
            if (result.equals(defaultHeader + defaultHeaderEnd)) {
                result = errorInfo;
            }
        } catch (Exception e){
            LOGGER.error("***报错堆栈截取报错***",e);
            result = errorInfo;
        }
        return result;
    }

    private static void appendSqlErrorInfo(StringBuilder result,String errorInfo){
        String keySentences = "org.hibernate.exception.SQLGrammarException: sql:\n";
        //堆栈内容
        int ignoreIndex = errorInfo.indexOf(keySentences);
        if(ignoreIndex != -1){
            int endIndex = errorInfo.indexOf("could not extract ResultSet");
            if(endIndex != -1){
                String head = errorInfo.substring(ignoreIndex + keySentences.length(),endIndex);
                result.append(head).append("=====\n");
            }
        }
        //截取结尾部分
        ignoreIndex = errorInfo.lastIndexOf("Caused by:");
        if(ignoreIndex != -1){
            result.append(errorInfo.substring(ignoreIndex));
        }
    }

    private static void appendLogicErrorInfo(StringBuilder result,String errorInfo){
        String keySentences = "com.af.expression.ExpressionException: ";
        //堆栈内容
        int ignoreIndex = errorInfo.indexOf(keySentences);
        if(ignoreIndex != -1){
            errorInfo = errorInfo.substring(ignoreIndex + keySentences.length());
            int ignoreIndexEnd = errorInfo.indexOf("at ");
            if(ignoreIndexEnd != -1){
                result.append(errorInfo, 0, ignoreIndexEnd).append("\n=====\n");
                errorInfo = errorInfo.substring(ignoreIndexEnd + 3);
                appendLogicErrorInfo(result, errorInfo);
            }
        } else {
            //截取结尾部分
            ignoreIndex = errorInfo.lastIndexOf("Caused by:");
            if(ignoreIndex != -1){
                result.append(errorInfo.substring(ignoreIndex));
            }
        }
    }
}
