package com.aote.plugins;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.*;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

/**
 * 操作文件
 */
public class FileTool {
    /**
     * 获取文件子目录
     */
    public JSONArray getChild(String rootPath) {
        rootPath = rootPath.trim().replace("\\", "/").replace("/", File.separator);
        JSONArray fileArr = new JSONArray();
        File file = new File(rootPath);
        File[] childArr = file.listFiles();
        for (File child : childArr) {
            JSONObject json = new JSONObject();
            //判断是否为文件夹
            if (child.isDirectory()) {
                json.put("size", 1);        // 如果是目录 则是1 , 文件 0
                json.put("suffix", "");
                json.put("type", "文件夹");
                json.put("name", child.getName());
            } else {
                //文件可能不存在后缀名（名称里面不存在" . "符号）
                if (child.getName().indexOf(".") == -1) {
                    json.put("suffix", "");
                    json.put("name", child.getName());
                } else {
                    json.put("suffix", child.getName().substring(child.getName().lastIndexOf(".") + 1));
                    json.put("name", child.getName().substring(0, child.getName().lastIndexOf(".")));
                }
                json.put("size", 0);            // 如果是目录 则是1 , 文件 0
                json.put("type", "文件");
            }
            //该文件路径，设key为id，用于打开文件用
            json.put("id", child.getPath());
            fileArr.put(json);
        }
        fileArr = jsonArrOrder(fileArr);
        return fileArr;
    }

    /**
     * 对文件排序
     * 文件夹优先，其次按照名称的字符序列排序，小的在前
     */
    private JSONArray jsonArrOrder(JSONArray data) {
        JSONArray sortedJsonArray = new JSONArray();
        List<JSONObject> jsonValues = new ArrayList<>();
        for (int i = 0; i < data.length(); i++) {
            jsonValues.add(data.getJSONObject(i));
        }
        Collections.sort(jsonValues, new Comparator<JSONObject>() {
            @Override
            public int compare(JSONObject a, JSONObject b) {
                if (!a.getString("type").equals(b.getString("type"))) {
                    if (a.getString("type").equals("文件夹"))
                        return -1;
                    return 1;
                }
                return a.getString("name").compareTo(b.getString("name"));
            }
        });
        for (int i = 0; i < data.length(); i++) {
            sortedJsonArray.put(jsonValues.get(i));
        }
        return sortedJsonArray;
    }

    /**
     * 删除指定路径的所有文件，包含本文件
     */
    public boolean deletetByPath(String path) {
        path = path.trim().replace("\\", "/").replace("/", File.separator);
        File file = new File(path);
        //该文件不存在直接返回false
        if (!file.exists()) {
            return false;
        }
        //文件不是文件夹时直接删除
        if (!file.isDirectory()) {
            return file.delete();
        }
        File[] childList = file.listFiles();
        //文件夹为空时直接删除
        if (childList.length == 0) {
            return file.delete();
        }
        //删除文件夹内的文件
        for (File p : childList) {
            if (!deletetByPath(p.getPath())) {
                return false;
            }
        }
        return file.delete();
    }

    /**
     * 通过指定路径创建文件夹
     */
    public boolean createFolderByPath(String path) {
        path = path.trim().replace("\\", "/").replace("/", File.separator);
        File file = new File(path);
        //如果文件夹不存在
        if (!file.exists()) {
            //创建文件夹
            file.mkdirs();
            return file.exists();
        }
        return false;
    }

    /**
     * 通过指定路径创建文件
     */
    public boolean createFileByPath(String path) {
        path = path.trim().replace("\\", "/").replace("/", File.separator);
        //当该文件不存在时创建
        File file = new File(path);
        if (!file.exists()) {
            try {
                BufferedWriter bw = new BufferedWriter(new FileWriter(path));
                bw.close();
                return file.exists();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return false;
    }

    /**
     * 通过指定路径修改文件名称
     */
    public boolean changeNameByPath(String path, String newName) {
        //检查文件新名称是否错误
        path = path.trim().replace("\\", "/").replace("/", File.separator);
        newName = newName.trim();
        if ("".equals(newName) || newName == null || newName.indexOf("/") != -1 || newName.indexOf("\\") != -1) {
            return false;
        }
        //检查原文件是否存在
        File file = new File(path);
        if (!file.exists()) {
            return false;
        }
        //组织新文件名称路径
        newName = path.substring(0, path.lastIndexOf(File.separator) + 1) + newName;
        File newFile = new File(newName);
        //判断新文件路径所表示的文件是否已经存在
        if (newFile.exists()) {
            return false;
        }
        try {
            //修改名称
            file.renameTo(newFile);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return true;
    }

    /**
     * 复制文件
     *
     * @param sourcePath 原文件路径
     * @param targetPath 粘贴文件路径
     */
    public boolean copyByPath(String sourcePath, String targetPath) {
        sourcePath = sourcePath.trim().replace("\\", "/").replace("/", File.separator);
        targetPath = targetPath.trim().replace("\\", "/").replace("/", File.separator);
        try {
            //获取源文件
            File sourceFile = new File(sourcePath);
            if (!sourceFile.exists()) {
                return false;
            }
            // 创建目标文件夹
            new File(targetPath).mkdirs();
            if (sourceFile.isFile()) {
                // 目标文件
                File targetFile = new File(targetPath + File.separator + sourceFile.getName());
                // 复制文件
                copyFile(sourceFile, targetFile);
            }
            if (sourceFile.isDirectory()) {
                // 复制目录
                String targetDir = targetPath + File.separator + sourceFile.getName();
                copyDirectiory(sourcePath, targetDir);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return true;
    }

    // 复制文件
    private static void copyFile(File sourceFile, File targetFile) throws IOException {
        // 新建文件输入流并对它进行缓冲
        FileInputStream input = new FileInputStream(sourceFile);
        BufferedInputStream inBuff = new BufferedInputStream(input);
        // 新建文件输出流并对它进行缓冲
        FileOutputStream output = new FileOutputStream(targetFile);
        BufferedOutputStream outBuff = new BufferedOutputStream(output);
        // 缓冲数组
        byte[] b = new byte[1024 * 5];
        int len;
        while ((len = inBuff.read(b)) != -1) {
            outBuff.write(b, 0, len);
        }
        // 刷新此缓冲的输出流
        outBuff.flush();
        //关闭流
        inBuff.close();
        outBuff.close();
        output.close();
        input.close();
    }

    // 复制文件夹
    private static void copyDirectiory(String sourceDir, String targetDir) throws IOException {
        // 创建目标目录
        new File(targetDir).mkdirs();
        // 获取源文件夹当前下的文件或目录
        File[] sourceFile = new File(sourceDir).listFiles();
        for (int i = 0; i < sourceFile.length; i++) {
            if (sourceFile[i].isFile()) {
                // 目标文件
                File targetFile = new File(new File(targetDir).getAbsolutePath() + File.separator + sourceFile[i].getName());
                // 复制文件
                copyFile(sourceFile[i], targetFile);
            }
            if (sourceFile[i].isDirectory()) {
                // 准备复制的源文件夹
                String dir1 = sourceDir + File.separator + sourceFile[i].getName();
                // 准备复制的目标文件夹
                String dir2 = targetDir + File.separator + sourceFile[i].getName();
                copyDirectiory(dir1, dir2);
            }
        }
    }

}
