package com.af.plugins.kafka.kafka_producer;

import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Random;

/**
 * @author 王泽松
 * @date 2019-5-9 pm 2:30
 * */
public final class KafkaProducerSingleton {

    private static final Logger LOGGER = LoggerFactory.getLogger(KafkaProducerSingleton.class);

    private static KafkaProducer<String, String> kafkaProducer;

    private Random random = new Random();

    private String topic;

    private int retry;

    private KafkaProducerSingleton() {

    }

    /**
     * 静态内部类
     *
     * @author tanjie
     *
     */
    private static class LazyHandler {
        private static final KafkaProducerSingleton instance = new KafkaProducerSingleton();
    }

    /**
     * 单例模式,kafkaProducer是线程安全的,可以多线程共享一个实例
     *
     * @return
     */
    public static KafkaProducerSingleton getInstance() {
        return LazyHandler.instance;
    }

    /**
     * kafka生产者进行初始化
     *
     * @return KafkaProducer
     */
    public void init(String topic,int retry) {
        //话题名称
        this.topic = topic;
        //发送消息失败的时候的重试次数
        this.retry = retry;
        //排除空指针异常，实例为空的时候初始化实例
        if (null == kafkaProducer) {
            //构建配置文件对象
            Properties prop = new Properties();
            try {
                prop.load(KafkaProducerSingleton.class.getResourceAsStream("/kafka_producer.properties"));
                //初始化kafka生产者实例对象
                kafkaProducer = new KafkaProducer<>(prop);
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 通过kafkaProducer发送消息
     *
     * @param topic
     *            消息接收主题
     * @param partitionNum
     *            哪一个分区
//     * @param retry
     *            重试次数
     * @param message
     *            具体消息值
     */
    public void sendKafkaMessage(String topic, int partitionNum, String message) {

        /*
         * 1、如果指定了某个分区,会只讲消息发到这个分区上
         * 2、如果同时指定了某个分区和key,则也会将消息发送到指定分区上,key不起作用
         * 3、如果没有指定分区和key,那么将会随机发送到topic的分区中
         * 4、如果指定了key,那么将会以hash<key>的方式发送到分区中
         * 5、发送消息的时候要附带对应业务的logic名字，数据和logic名字用**号隔开
         */
        ProducerRecord<String, String> record = new ProducerRecord<String, String>(
                topic, random.nextInt(partitionNum), "", message);
        // send方法是异步的,添加消息到缓存区等待发送,并立即返回，这使生产者通过批量发送消息来提高效率
        // kafka生产者是线程安全的,可以单实例发送消息
        kafkaProducer.send(record, (recordMetadata, exception) -> {
            if (null != exception) {
                LOGGER.error("kafka发送消息失败:" + exception.getMessage(),
                        exception);
                //重试发送消息
                retryKakfaMessage(topic, partitionNum, message);
            }
        });
//        kafkaProducer.close();
    }

    /**
     * 当kafka消息发送失败后,重试
     *
     * @param retryMessage
     */
    private void retryKakfaMessage(final String retryTopic,final int retryPartitionNum, final String retryMessage) {
        ProducerRecord<String, String> record = new ProducerRecord<>(
                retryTopic, retryPartitionNum, "", retryMessage);
        for (int i = 1; i <= retry; i++) {
            try {
                //重新发送消息
                kafkaProducer.send(record);
                return;
            } catch (Exception e) {
                LOGGER.error("kafka发送消息失败:" + e.getMessage(), e);
                //递归调用重新发送消息
                retryKakfaMessage(retryTopic,retryPartitionNum,retryMessage);
            }
        }
    }

    /**
     * kafka实例销毁
     */
    public void close() {
        if (null != kafkaProducer) {
            kafkaProducer.close();
        }
    }

    public String getTopic() {
        return topic;
    }

    public void setTopic(String topic) {
        this.topic = topic;
    }

    public int getRetry() {
        return retry;
    }

    public void setRetry(int retry) {
        this.retry = retry;
    }

}