package com.af.plugins;

import org.apache.http.*;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.config.SocketConfig;
import org.apache.http.conn.ConnectionKeepAliveStrategy;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.LayeredConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.*;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicHeaderElementIterator;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.HTTP;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.util.EntityUtils;
import org.apache.log4j.Logger;
import org.json.JSONObject;

import javax.net.ssl.*;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.URI;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * 新Rest请求连接工具
 * 高可用连接池
 * @author Mrriver
 */
public class HttpConnectionPoolUtil {

    private static final Logger logger = Logger.getLogger(HttpConnectionPoolUtil.class);

    /**
     *设置连接建立的超时时间
     */
    private static final int CONNECT_TIMEOUT = Config.getConnectTimeout();
    private static final int SOCKET_TIMEOUT = Config.getConnectTimeout();
    /**
     * 最大连接数
     */
    private static final int MAX_CONN = Config.getHttpMaxPoolSize();
    /**
     * 发送HTTP请求的客户端单例
     */
    private static CloseableHttpClient httpClient;
    /**
     * 相当于线程锁,用于线程安全
     */
    private final static Object syncLock = new Object();

    /**
     * 对http请求进行基本设置
     * @param httpRequestBase http请求
     */
    private static void setRequestConfig(HttpRequestBase httpRequestBase){
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(CONNECT_TIMEOUT)
                .setConnectTimeout(CONNECT_TIMEOUT)
                .setSocketTimeout(SOCKET_TIMEOUT).build();
        httpRequestBase.setConfig(requestConfig);
    }

    /**
     * 获取HTTP请求 httpclient实例
     * @return httpclient实例
     */
    public static CloseableHttpClient getHttpClient() {
        if (httpClient == null){
            //多线程下同时调用getHttpClient容易导致重复创建httpClient对象的问题,所以加上了同步锁
            synchronized (syncLock){
                if (httpClient == null){
                    httpClient = createHttpClient();
                }
            }
        }
        return httpClient;
    }

    /**
     * 获取HTTPS请求 httpclient实例（SSL证书）
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    public static CloseableHttpClient getHttpClient(LayeredConnectionSocketFactory sslSocketFactory) {
        return createHttpClient(sslSocketFactory);
    }

    /**
     * 构建httpclient实例
     * @return httpclient实例
     */
    private static CloseableHttpClient createHttpClient() {
        return createHttpClient(null);
    }

    /**
     * 构建httpclient实例
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    private static CloseableHttpClient createHttpClient(LayeredConnectionSocketFactory sslSocketFactory) {
        //设置https相关信息
        if(sslSocketFactory == null){
            logger.debug("创建http客户端会话");
            sslSocketFactory = SSLConnectionSocketFactory.getSocketFactory();
        } else {
            logger.debug("创建https客户端会话");
        }
        Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory> create()
                .register("http", PlainConnectionSocketFactory.INSTANCE)
                .register("https", sslSocketFactory).build();

        PoolingHttpClientConnectionManager manager = new PoolingHttpClientConnectionManager(registry);
        //设置连接参数
        // 最大连接数
        manager.setMaxTotal(MAX_CONN);
        // 路由最大连接数
        manager.setDefaultMaxPerRoute(MAX_CONN);

        //请求失败时,进行请求重试
        HttpRequestRetryHandler handler = (e, i, httpContext) -> {
            if (i > 3){
                //重试超过3次,放弃请求
                logger.error("重试超过3次,放弃请求");
                return false;
            }
            if (e instanceof NoHttpResponseException){
                logger.error("服务器没有响应,可能是服务器断开了连接,进行重试");
                return true;
            }
            if (e instanceof SSLHandshakeException){
                logger.error("SSL握手异常:",e);
                return false;
            }
            if (e instanceof InterruptedIOException){
                //超时
                logger.error("HTTP请求连接超时,进行重试");
                return true;
            }
            if (e instanceof UnknownHostException){
                logger.error("服务器不可达:",e);
                return false;
            }
            if (e instanceof SSLException){
                logger.error("SSL异常:",e);
                return false;
            }

            HttpClientContext context = HttpClientContext.adapt(httpContext);
            HttpRequest request = context.getRequest();
            //如果请求不是关闭连接的请求
            return !(request instanceof HttpEntityEnclosingRequest);
        };
        //设置keepalive
        ConnectionKeepAliveStrategy myStrategy = (response, context) -> {
            // Honor 'keep-alive' header
            HeaderElementIterator it = new BasicHeaderElementIterator(
                    response.headerIterator(HTTP.CONN_KEEP_ALIVE));
            while (it.hasNext()) {
                HeaderElement he = it.nextElement();
                String param = he.getName();
                String value = he.getValue();
                if (value != null && "timeout".equalsIgnoreCase(param)) {
                    try {
                        return Long.parseLong(value) * 1000;
                    } catch(NumberFormatException ignore) {
                    }
                }
            }
            HttpHost target = (HttpHost) context.getAttribute(
                    HttpClientContext.HTTP_TARGET_HOST);
            if ("www.naughty-server.com".equalsIgnoreCase(target.getHostName())) {
                // Keep alive for 5 seconds only
                return 5 * 1000;
            } else {
                // otherwise keep alive for 30 seconds
                return 30 * 1000;
            }
        };

        return HttpClients.custom()
                .setConnectionManager(manager)
                .setRetryHandler(handler)
                .setKeepAliveStrategy(myStrategy)
                .evictExpiredConnections()
                .evictIdleConnections(Config.getHttpIdelTimeout(),TimeUnit.MILLISECONDS)
                .build();
    }

    /**
     * 绕过HTTPS验证
     *
     * @return
     * @throws NoSuchAlgorithmException
     * @throws KeyManagementException
     */
    public static SSLContext createIgnoreVerifySSL() throws NoSuchAlgorithmException, KeyManagementException {
        // 信任所有证书
        SSLContext sslContext = null;
        try {
            sslContext = new SSLContextBuilder().loadTrustMaterial(null, (TrustStrategy) (chain, authType) -> true).build();
        } catch (KeyStoreException e) {
            e.printStackTrace();
        }
        return sslContext;
    }

    /**
     * 设置请求体
     * @param base 基础客户端对象
     * @param params 请求体字符串
     */
    private static void setPostParams(HttpEntityEnclosingRequestBase base, String params){
        //设置请求体
        if(params != null && !"".equals(params)){
            StringEntity se = new StringEntity(params,StandardCharsets.UTF_8);
            base.setEntity(se);
        }
    }

    /**
     * 设置请求头
     * @param base 基础客户端对象
     * @param headersStr 请求头字符串
     */
    private static void setHeaders(HttpRequestBase base,String headersStr) {
        // 设置请求头
        if (headersStr != null && !"".equals(headersStr)) {
            JSONObject headers = new JSONObject(headersStr);
            Iterator<String> keys = headers.keys();
            while (keys.hasNext()) {
                String key = keys.next();
                String val = headers.getString(key);
                base.setHeader(key, val);
            }
        }
    }

    /**
     * 发送通用Http请求
     * @param path  请求路径
     * @param value 请求参数
     * @param headersStr 请求头
     * @param base  请求类型
     * @return  请求结果
     */
    public static String request(String path, String value, String headersStr, HttpRequestBase base) throws IOException {
        return request(path,value,headersStr,base,getHttpClient());
    }

    /**
     * 发送通用Http请求
     * @param path  请求路径
     * @param value 请求参数
     * @param headersStr 请求头
     * @param base  请求类型
     * @param httpClient client对象
     * @return  请求结果
     */
    public static String request(String path,
                                        String value,
                                        String headersStr,
                                        HttpRequestBase base,
                                        CloseableHttpClient httpClient) throws IOException {
        //配置请求参数
        setRequestConfig(base);
        //设置请求体
        if(base instanceof HttpEntityEnclosingRequestBase) {
            setPostParams((HttpEntityEnclosingRequestBase) base, value);
        }
        //设置请求头
        setHeaders(base,headersStr);
        //设置请求地址
        base.setURI(URI.create(path));
        CloseableHttpResponse response = null;
        String result;
        try {
            long begin = System.currentTimeMillis();
            response = httpClient.execute(base, HttpClientContext.create());
            long end = System.currentTimeMillis();
            long time = end - begin;
            String text = "请求接口耗时："+time+"ms";
            if(time >= 500){
                logger.warn(text);
            } else {
                logger.info(text);
            }
            String code = String.valueOf(response.getStatusLine().getStatusCode());
            // 获取数据成功，返回数据
            if (code.startsWith("2")) {
                HttpEntity entity = response.getEntity();
                if(entity != null){
                    result = EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8);
                } else {
                    return null;
                }
            } else {
                String errorEntity;
                String data = response.getStatusLine().getReasonPhrase();
                HttpEntity entity = response.getEntity();
                if(entity != null){
                    errorEntity = EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8);
                } else {
                    errorEntity = "";
                }
                // 返回错误码
                result = "{status: " + code + ", data: '" + data + "', errorEntity: '"+ errorEntity +"'}";
            }
        } finally {
            if (response != null) {
                response.close();
            }
        }
        return result;
    }

    /**
     * 关闭连接池
     */
    public static void closeConnectionPool(CloseableHttpClient httpClient,PoolingHttpClientConnectionManager manager){
        try {
            httpClient.close();
            manager.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public static String requestFormUrlEncoded(String path,
                                               JSONObject params,
                                               HttpEntityEnclosingRequestBase base,
                                               CloseableHttpClient client) throws IOException {
        //配置请求参数
        setRequestConfig(base);
        //设置请求地址
        base.setURI(URI.create(path));
        //设置请求体
        base.setEntity(new UrlEncodedFormEntity(paramsConverter(params)));
        CloseableHttpResponse response = null;
        String result = null;
        try {
            long begin = System.currentTimeMillis();
            response = client.execute(base, HttpClientContext.create());
            long end = System.currentTimeMillis();
            logger.warn("请求接口耗时："+(end-begin)+"ms");
            result = EntityUtils.toString(response.getEntity(),StandardCharsets.UTF_8);
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (response != null) {
                response.close();
            }
        }
        return result;
    }

    public static String requestFormUrlEncoded(String path,
                                               Map<String,String> params,
                                               HttpEntityEnclosingRequestBase base,
                                               CloseableHttpClient client) throws IOException {
        return requestFormUrlEncoded(path, new JSONObject(params), base, client);
    }

    private static List<NameValuePair> paramsConverter(Map<String, String> params) {
        List<NameValuePair> nvps = new LinkedList<>();
        Set<Map.Entry<String, String>> paramsSet = params.entrySet();
        for (Map.Entry<String, String> paramEntry : paramsSet) {
            nvps.add(new BasicNameValuePair(paramEntry.getKey(), paramEntry
                    .getValue()));
        }
        return nvps;
    }

    private static List<NameValuePair> paramsConverter(JSONObject params) {
        List<NameValuePair> nvps = new LinkedList<>();
        Set<String> paramsSet = params.keySet();
        for (String key : paramsSet) {
            nvps.add(new BasicNameValuePair(key, params.getString(key)));
        }
        return nvps;
    }
}
