package com.af.plugins;

import org.apache.http.*;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.ConnectionKeepAliveStrategy;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.impl.nio.client.HttpAsyncClients;
import org.apache.http.impl.nio.conn.PoolingNHttpClientConnectionManager;
import org.apache.http.impl.nio.reactor.DefaultConnectingIOReactor;
import org.apache.http.impl.nio.reactor.IOReactorConfig;
import org.apache.http.message.BasicHeaderElementIterator;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.nio.conn.NoopIOSessionStrategy;
import org.apache.http.nio.conn.SchemeIOSessionStrategy;
import org.apache.http.nio.conn.ssl.SSLIOSessionStrategy;
import org.apache.http.nio.reactor.ConnectingIOReactor;
import org.apache.http.nio.reactor.IOReactorException;
import org.apache.http.protocol.HTTP;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.util.EntityUtils;
import org.apache.log4j.Logger;
import org.json.JSONObject;

import javax.net.ssl.SSLContext;
import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.*;

/**
 * 新Rest异步请求连接工具
 * 高可用连接池
 * @author Mrriver
 */
public class HttpAsyncConnectionPoolUtil {

    private static final Logger logger = Logger.getLogger(HttpAsyncConnectionPoolUtil.class);

    /**
     *设置连接建立的超时时间
     */
    private static final int CONNECT_TIMEOUT = Config.getConnectTimeout();
    private static final int SOCKET_TIMEOUT = Config.getConnectTimeout();
    /**
     * 最大连接数
     */
    private static final int MAX_CONN = Config.getHttpMaxPoolSize();
    /**
     * 发送HTTP异步请求的客户端单例
     */
    private static CloseableHttpAsyncClient httpClient;
    /**
     * 相当于线程锁,用于线程安全
     */
    private final static Object syncLock = new Object();

    /**
     * 对HTTP异步请求进行基本设置
     * @param httpRequestBase HTTP异步请求
     */
    private static void setRequestConfig(HttpRequestBase httpRequestBase){
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(CONNECT_TIMEOUT)
                .setConnectTimeout(CONNECT_TIMEOUT)
                .setSocketTimeout(SOCKET_TIMEOUT).build();
        httpRequestBase.setConfig(requestConfig);
    }

    /**
     * 获取HTTP异步请求 httpclient实例
     * @return httpclient实例
     */
    public static CloseableHttpAsyncClient getHttpClient() {
        if (httpClient == null){
            //多线程下同时调用getHttpClient容易导致重复创建httpClient对象的问题,所以加上了同步锁
            synchronized (syncLock){
                if (httpClient == null){
                    httpClient = createHttpClient();
                }
            }
        }
        httpClient.start();
        return httpClient;
    }

    /**
     * 获取HTTPS异步请求 httpclient实例（SSL证书）
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    public static CloseableHttpAsyncClient getHttpClient(SSLIOSessionStrategy sslSocketFactory) {
        return createHttpClient(sslSocketFactory);
    }

    /**
     * 构建httpclient实例
     * @return httpclient实例
     */
    private static CloseableHttpAsyncClient createHttpClient() {
        return createHttpClient(null);
    }

    /**
     * 构建httpclient实例
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    private static CloseableHttpAsyncClient createHttpClient(SSLIOSessionStrategy sslSocketFactory) {
        //设置https相关信息
        if(sslSocketFactory == null){
            logger.debug("创建HTTP异步客户端会话");
            sslSocketFactory = SSLIOSessionStrategy.getDefaultStrategy();
        } else {
            logger.debug("创建https异步客户端会话");
        }
        Registry<SchemeIOSessionStrategy> registry = RegistryBuilder.<SchemeIOSessionStrategy>create()
                .register("http", NoopIOSessionStrategy.INSTANCE)
                .register("https", sslSocketFactory)
                .build();

        // 配置io线程
        IOReactorConfig ioReactorConfig = IOReactorConfig.custom().
                setIoThreadCount(Runtime.getRuntime().availableProcessors())
                .setSoKeepAlive(true)
                .build();
        // 设置连接池大小
        ConnectingIOReactor ioReactor = null;
        try {
            ioReactor = new DefaultConnectingIOReactor(ioReactorConfig);
        } catch (IOReactorException e) {
            e.printStackTrace();
        }
        PoolingNHttpClientConnectionManager manager = new PoolingNHttpClientConnectionManager(ioReactor,registry);
        //设置连接参数
        // 最大连接数
        manager.setMaxTotal(MAX_CONN);
        // 路由最大连接数
        manager.setDefaultMaxPerRoute(MAX_CONN);

        //设置keepalive
        ConnectionKeepAliveStrategy myStrategy = (response, context) -> {
            // Honor 'keep-alive' header
            HeaderElementIterator it = new BasicHeaderElementIterator(
                    response.headerIterator(HTTP.CONN_KEEP_ALIVE));
            while (it.hasNext()) {
                HeaderElement he = it.nextElement();
                String param = he.getName();
                String value = he.getValue();
                if (value != null && "timeout".equalsIgnoreCase(param)) {
                    try {
                        return Long.parseLong(value) * 1000;
                    } catch(NumberFormatException ignore) {
                    }
                }
            }
            HttpHost target = (HttpHost) context.getAttribute(
                    HttpClientContext.HTTP_TARGET_HOST);
            if ("www.naughty-server.com".equalsIgnoreCase(target.getHostName())) {
                // Keep alive for 5 seconds only
                return 5 * 1000;
            } else {
                // otherwise keep alive for 30 seconds
                return 30 * 1000;
            }
        };

        return HttpAsyncClients.custom()
                .setConnectionManager(manager)
                .setKeepAliveStrategy(myStrategy)
                .build();
    }

    /**
     * 绕过HTTPS验证
     *
     * @return
     * @throws NoSuchAlgorithmException
     * @throws KeyManagementException
     */
    public static SSLContext createIgnoreVerifySSL() throws NoSuchAlgorithmException, KeyManagementException {
        // 信任所有证书
        SSLContext sslContext = null;
        try {
            sslContext = new SSLContextBuilder().loadTrustMaterial(null, (TrustStrategy) (chain, authType) -> true).build();
        } catch (KeyStoreException e) {
            e.printStackTrace();
        }
        return sslContext;
    }

    /**
     * 设置请求体
     * @param base 基础客户端对象
     * @param params 请求体字符串
     */
    private static void setPostParams(HttpEntityEnclosingRequestBase base, String params){
        //设置请求体
        if(params != null && !"".equals(params)){
            StringEntity se = new StringEntity(params,StandardCharsets.UTF_8);
            base.setEntity(se);
        }
    }

    /**
     * 设置请求头
     * @param base 基础客户端对象
     * @param headersStr 请求头字符串
     */
    private static void setHeaders(HttpRequestBase base,String headersStr) {
        // 设置请求头
        if (headersStr != null && !"".equals(headersStr)) {
            JSONObject headers = new JSONObject(headersStr);
            Iterator<String> keys = headers.keys();
            while (keys.hasNext()) {
                String key = keys.next();
                String val = headers.getString(key);
                base.setHeader(key, val);
            }
        }
    }

    /**
     * 发送通用HTTP异步请求
     * @param path  请求路径
     * @param value 请求参数
     * @param headersStr 请求头
     * @param base  请求类型
     */
    public static void request(String path, String value, String headersStr, HttpRequestBase base) throws IOException {
        request(path,value,headersStr,base,getHttpClient());
    }

    /**
     * 发送通用HTTP异步请求
     * @param path  请求路径
     * @param value 请求参数
     * @param headersStr 请求头
     * @param base  请求类型
     * @param httpClient client对象
     */
    public static void request(String path,
                                        String value,
                                        String headersStr,
                                        HttpRequestBase base,
                                        CloseableHttpAsyncClient httpClient) {
        //配置请求参数
        setRequestConfig(base);
        //设置请求体
        if(base instanceof HttpEntityEnclosingRequestBase) {
            setPostParams((HttpEntityEnclosingRequestBase) base, value);
        }
        //设置请求头
        setHeaders(base,headersStr);
        //设置请求地址
        base.setURI(URI.create(path));
        long begin = System.currentTimeMillis();
        httpClient.execute(base, new InnerAsyncResponse());
        long end = System.currentTimeMillis();
        long time = end - begin;
        String text = "请求接口耗时："+time+"ms";
        if(time >= 500){
            System.out.println(text);
        } else {
            System.out.println(text);
        }
    }

    /**
     * 关闭连接池
     */
    public static void closeConnectionPool(CloseableHttpAsyncClient httpClient){
        try {
            httpClient.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public static void requestFormUrlEncoded(String path,
                                               JSONObject params,
                                               HttpEntityEnclosingRequestBase base,
                                               CloseableHttpAsyncClient httpClient) throws IOException {
        //配置请求参数
        setRequestConfig(base);
        //设置请求地址
        base.setURI(URI.create(path));
        //设置请求体
        base.setEntity(new UrlEncodedFormEntity(paramsConverter(params)));
        long begin = System.currentTimeMillis();
        httpClient.execute(base, new InnerAsyncResponse());
        long end = System.currentTimeMillis();
        logger.warn("请求接口耗时："+(end-begin)+"ms");
    }

    public static void requestFormUrlEncoded(String path,
                                             Map<String,String> params,
                                             HttpEntityEnclosingRequestBase base,
                                             CloseableHttpAsyncClient httpClient) throws IOException {
        requestFormUrlEncoded(path,new JSONObject(params),base,httpClient);
    }

    private static List<NameValuePair> paramsConverter(JSONObject params) {
        List<NameValuePair> nvps = new LinkedList<>();
        Set<String> paramsSet = params.keySet();
        for (String key : paramsSet) {
            nvps.add(new BasicNameValuePair(key, params.getString(key)));
        }
        return nvps;
    }

    public static class InnerAsyncResponse implements FutureCallback<HttpResponse> {

        @Override
        public void completed(HttpResponse result) {
            String resStr = null;
            String code = String.valueOf(result.getStatusLine().getStatusCode());
            // 获取数据成功，返回数据
            if (code.startsWith("2")) {
                HttpEntity entity = result.getEntity();
                if(entity != null){
                    try {
                        resStr = EntityUtils.toString(result.getEntity(), StandardCharsets.UTF_8);
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
            } else {
                String errorEntity = null;
                String data = result.getStatusLine().getReasonPhrase();
                HttpEntity entity = result.getEntity();
                if(entity != null){
                    try {
                        errorEntity = EntityUtils.toString(result.getEntity(), StandardCharsets.UTF_8);
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                } else {
                    errorEntity = "";
                }
                // 返回错误码
                resStr = "{status: " + code + ", data: '" + data + "', errorEntity: '"+ errorEntity +"'}";
            }
            System.out.println("异步请求结果：" + resStr);
        }

        @Override
        public void failed(Exception ex) {
            throw new RuntimeException(ex);
        }

        @Override
        public void cancelled() {

        }
    }
}
