package com.af.plugins;

import com.alibaba.fastjson.JSON;
import org.json.JSONArray;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.Set;

/**
 * json工具类
 *
 * @author zt
 *
 */
public class JsonTools {

	/**
	 * 给一个JSONObject中添加对象
	 * @param obj  json对象
	 * @param addLabel 键
	 * @param addVale 值
	 * @return 添加完成后返回obj
	 */
	public static JSONObject add(JSONObject obj, String addLabel, Object addVale) {
		obj.put(addLabel, addVale);
		return obj;
	}

	/**
	 * 给一个JSONObject追加一个JSONObject中的所有值
	 *
	 * @param obj    被追加的JSONObject
	 * @param addObj 追加的JSONObject
	 * @return 追加完的JSONObject
	 */
	public static JSONObject addJSON(JSONObject obj, JSONObject addObj) {
		if (addObj == null) {
			throw new NullPointerException("追加的JSONObject不存在！");
		}
		Set<String> keys = addObj.keySet();
		for (String key : keys) {
			obj.put(key, addObj.get(key));
		}
		return obj;
	}

	/**
	 * 判断JSONObject对象中是否包含key
	 * @param obj  json对象
	 * @param key  是否包含的key键
	 * @return boolean值的返回值
	 */
	public static boolean isContains(JSONObject obj, String key) {
		return obj.has(key);
	}

	/**
	 * JSONObject中key是否为空
	 * @param obj json对象
	 * @param key 需要查看的key
	 * @return 返回的boolean值
	 */
	public static boolean isNull(JSONObject obj, String key) {
		return obj.isNull(key);
	}

	/**
	 * JSONObject是否为空
	 * @param obj  查看的json对象
	 * @return 返回boolean
	 */
	public static boolean isNull(JSONObject obj) {
		return !obj.isEmpty();
	}

	/**
	 * 将字符串转化为json
	 *
	 * @param str 要转为json的字符串
	 * @return 原生json
	 */
	public static JSONObject convertToJson(String str) {
		if (str == null || "".equals(str)) {
			return new JSONObject();
		}
		return new JSONObject(str);
	}

	/**
	 * 将字符串转化为FastJson的json
	 *
	 * @param str 要转为json的字符串
	 * @return FastJson的json
	 */
	public static com.alibaba.fastjson.JSONObject convertToFastJson(String str) {
		if(str == null || "".equals(str)){
			return new com.alibaba.fastjson.JSONObject();
		}
		return JSON.parseObject(str);
	}

	/**
	 * 把字符串转换成JSON数组
	 * @param str 需要转换的String
	 * @return 转换完成后的json数组
	 */
	public static JSONArray parseArray(String str) {
		return new JSONArray(str);
	}

	/**
	 * 返回一个JSON数组实例
	 *
	 * @return JSON数组实例
	 */
	public static JSONArray getArray() {
		return new JSONArray();
	}

	/**
	 * 返回一个制定长度的JSON数组实例
	 *
	 * @param val 长度
	 * @return
	 */
	public static JSONArray getArray(Object val) {
		int len;
		try {
			len = new BigDecimal(val.toString()).intValue();
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		return new JSONArray(new String[len]);
	}

	/**
	 * 返回一个JSON实例
	 *
	 * @return JSON数组实例
	 */
	public static JSONObject getJson() {return new JSONObject();}

	/**
	 * 这个方法是将json转换成Map
	 * @param object json对象
	 * @return 转换后的map
	 */
	public static Map<String, Object> toHashMap(JSONObject object) {
		return object.toMap();
	}

	/**
	 * 应用场景：对于售气收费，为匹配气价的计算，将json字符串进行相应的包装
	 * 未包装前:{"data":{"price":"1","amount":"12"}}
	 * 包装后:{"data":[{"price":"1","amount":"12"}]}
	 *
	 * @param json json对象
	 * @param pricetype 气价类型
	 * @return 包装后的json数组
	 */
	public static JSONArray packJson(JSONObject json, String pricetype) {
		JSONArray result = new JSONArray();
		if ("fixed".equals(pricetype)) {
			JSONObject value = new JSONObject();
			value.put("f_unitprice", new BigDecimal(String.valueOf(json.get("f_unitprice"))));
			result.put(value);
		} else if ("mixed".equals(pricetype)) {
			JSONArray array = json.getJSONArray("detailprice");
			JSONObject obj = new JSONObject();
			obj.put("f_hybridprice1",
					new BigDecimal(String.valueOf(array.getJSONObject(0).get("f_price"))));
			obj.put("f_hybridprice2",
					new BigDecimal(String.valueOf(array.getJSONObject(1).get("f_price"))));
			obj.put("f_hybridprice1rate",
					new BigDecimal(String.valueOf(array.getJSONObject(0).get("f_ratio"))));
			result.put(obj);
		} else if ("staired".equals(pricetype)) {
			JSONArray array = json.getJSONArray("detailprice");
			for (Object object : array) {
				JSONObject obj = new JSONObject();
				obj.put("price", new BigDecimal(String.valueOf(((JSONObject) object).get("f_price"))));
				obj.put("amount",new BigDecimal(String.valueOf(((JSONObject) object).get("f_amount"))));
				result.put(obj);
			}
			JSONObject obj = new JSONObject();
			obj.put("sumamount", json.get("sumamount"));
			result.put(obj);
		}
		return result;
	}
	/**
	 * 读取JSON文件，返回JSON对象
	 * @param fileName 文件路径
	 * @return JSONObject
	 */
	public static JSONObject readJsonFile(String fileName) {
		BufferedReader reader = null;
		StringBuilder laststr = new StringBuilder();
		try {
			InputStream input = JsonTools.class
					.getResourceAsStream("/" + fileName);
			InputStreamReader inputStreamReader = new InputStreamReader(input,
					StandardCharsets.UTF_8);
			reader = new BufferedReader(inputStreamReader);
			String tempString;
			while ((tempString = reader.readLine()) != null) {
				laststr.append(tempString);
			}
			reader.close();
			return new JSONObject(laststr.toString());
		} catch (IOException e) {
			throw new RuntimeException(e);
		} finally {
			if (reader != null) {
				try {
					reader.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}

	/**
	 * 读取JSON文件，返回JSON数组
	 * @param fileName 文件路径
	 * @return JSONArray
	 */
	public static JSONArray readJsonArrayFile(String fileName) {
		BufferedReader reader = null;
		StringBuilder laststr = new StringBuilder();
		try {
			InputStream input = JsonTools.class
					.getResourceAsStream("/"+fileName);
			InputStreamReader inputStreamReader = new InputStreamReader(input,
					StandardCharsets.UTF_8);
			reader = new BufferedReader(inputStreamReader);
			String tempString;
			while ((tempString = reader.readLine()) != null) {
				laststr.append(tempString);
			}
			reader.close();
			return new JSONArray(laststr.toString());
		} catch (IOException e) {
			throw new RuntimeException(e);
		} finally {
			if (reader != null) {
				try {
					reader.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}

	/**
	 * JSON键值对替换（保留原JSON中的键值对）
	 * @param oldObj 被替换的JSON
	 * @param newObj 替换的JSON
	 * @return 处理后的JSON
	 */
	public static JSONObject replaceJSON(JSONObject oldObj,JSONObject newObj){
		for(String key : newObj.keySet()){
			//如果新JSON中有原JSON的键值对
			oldObj.put(key,newObj.get(key));
		}
		return oldObj;
	}
}
