package com.aote.sxxh;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Security;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.RSAPublicKeySpec;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;


import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

//import com.sxxh.pay.utils.Config;
//import com.sxxh.pay.utils.LogUtil;

public class CertUtil {
	
	static Logger log = Logger.getLogger(CertUtil.class);
	
	/** 证书容器. */
	private static KeyStore keyStore = null;

	/** 加密公钥 */
	private static PublicKey encryptTrackKey = null;

	/** 验证签名证书. */
	private static X509Certificate validateCert = null;

	/** 验签证书存储Map. */
	private static Map<String, X509Certificate> certMap = new HashMap<String, X509Certificate>();

	/** 根据传入证书文件路径和密码读取指定的证书容器.(一种线程安全的实现方式) */
	private final static ThreadLocal<KeyStore> certKeyStoreLocal = new ThreadLocal<KeyStore>();

	/** 基于Map存储多商户RSA私钥 */
	private final static Map<String, KeyStore> certKeyStoreMap = new ConcurrentHashMap<String, KeyStore>();



	

	/**
	 * 根据传入的证书文件路径和证书密码加载指定的签名证书
	 * @deprecated
	 */
	public static void initSignCert(String certFilePath, String certPwd) {
//		log.debug("加载证书文件[" + certFilePath + "]和证书密码[" + certPwd
//				+ "]的签名证书开始.");
		certKeyStoreLocal.remove();
		File files = new File(certFilePath);
		if (!files.exists()) {
			log.debug("证书文件不存在,初始化签名证书失败.");
			return;
		}
		try {
			certKeyStoreLocal.set(getKeyInfo(certFilePath, certPwd, "PKCS12"));
		} catch (IOException e) {
			log.debug("加载签名证书失败", e);
		}
		log.debug("加载证书文件[" + certFilePath + "]和证书密码[" + certPwd
				+ "]的签名证书结束.");
	}


	/**
	 * 加载RSA签名证书
	 * 
	 * @param certFilePath
	 * @param certPwd
	 */
	public static void loadRsaCert(String certFilePath, String certPwd) {
		KeyStore keyStore = null;
		try {
			keyStore = getKeyInfo(certFilePath, certPwd, "PKCS12");
			certKeyStoreMap.put(certFilePath, keyStore);
			log.debug("LoadRsaCert Successful");
		} catch (IOException e) {
			log.debug("LoadRsaCert Error", e);
		}
	}


	/**
	 * 从指定目录下加载验证签名证书
	 * 
	 */
	public static X509Certificate getValidateCert(String ValidateCertPath) {
		if(validateCert != null ){
			return validateCert;
		}
		
		log.debug("加载验证签名证书路径==>" + ValidateCertPath);
		if (StringUtils.isEmpty(ValidateCertPath)) {
			log.debug("ERROR: acpsdk.validateCert.dir is empty");
			return null;
		}
		File file = new File(ValidateCertPath);
		CertificateFactory cf = null;
		FileInputStream in = null;
		try {
			cf = CertificateFactory.getInstance("X.509");
			in = new FileInputStream(file.getAbsolutePath());
			validateCert = (X509Certificate) cf.generateCertificate(in);
			// 打印证书加载信息,供测试阶段调试
			log.debug("[" + file.getAbsolutePath() + "][CertId="
					+ validateCert.getSerialNumber().toString() + "]");
			log.debug("LoadVerifyCert Successful");
		} catch (CertificateException e) {
			log.debug("LoadVerifyCert Error", e);
		} catch (FileNotFoundException e) {
			log.debug("LoadVerifyCert Error File Not Found", e);
		} finally {
			if (null != in) {
				try {
					in.close();
				} catch (IOException e) {
					log.debug(e.toString());
				}
			}
		}
		return validateCert;
	}


	


	/**
	 * 通过传入证书绝对路径和证书密码获取所对应的签名证书私钥
	 * 
	 * @param certPath
	 *            证书绝对路径
	 * @param certPwd
	 *            证书密码
	 * @return 证书私钥
	 * 
	 * @deprecated
	 */
	public static PrivateKey getSignCertPrivateKeyByThreadLocal(
			String certPath, String certPwd) {
		if (null == certKeyStoreLocal.get()) {
			// 初始化指定certPath和certPwd的签名证书容器
			initSignCert(certPath, certPwd);
		}
		try {
			Enumeration<String> aliasenum = certKeyStoreLocal.get().aliases();
			String keyAlias = null;
			if (aliasenum.hasMoreElements()) {
				keyAlias = aliasenum.nextElement();
			}
			PrivateKey privateKey = (PrivateKey) certKeyStoreLocal.get()
					.getKey(keyAlias, certPwd.toCharArray());
			return privateKey;
		} catch (Exception e) {
			log.debug("获取[" + certPath + "]的签名证书的私钥失败", e);
			return null;
		}
	}

	public static PrivateKey getSignCertPrivateKeyByStoreMap(String certPath,
			String certPwd) {
		if (!certKeyStoreMap.containsKey(certPath)) {
			loadRsaCert(certPath, certPwd);
		}
		try {
			Enumeration<String> aliasenum = certKeyStoreMap.get(certPath)
					.aliases();
			String keyAlias = null;
			if (aliasenum.hasMoreElements()) {
				keyAlias = aliasenum.nextElement();
			}
			PrivateKey privateKey = (PrivateKey) certKeyStoreMap.get(certPath)
					.getKey(keyAlias, certPwd.toCharArray());
			return privateKey;
		} catch (KeyStoreException e) {
			log.debug("getSignCertPrivateKeyByStoreMap Error", e);
			return null;
		} catch (UnrecoverableKeyException e) {
			log.debug("getSignCertPrivateKeyByStoreMap Error", e);
			return null;
		} catch (NoSuchAlgorithmException e) {
			log.debug("getSignCertPrivateKeyByStoreMap Error", e);
			return null;
		}
	}


	/**
	 * 获取加密证书公钥.密码加密时需要
	 * 加密磁道信息证书
	 * 
	 * @return
	 */
	public static PublicKey getEncryptTrackPublicKey() {
		if (null == encryptTrackKey) {
			//initTrackKey();
		}
		return encryptTrackKey;
	}

	/**
	 * 验证签名证书
	 * 
	 * @return 验证签名证书的公钥
	 */
	public static PublicKey getValidateKey() {
		if (null != validateCert) {
			return validateCert.getPublicKey();
		}
		return validateCert.getPublicKey();
	}




	/**
	 * 获取签名证书中的证书序列号（单证书）考虑cfca颁发的符合证书
	 * 
	 * @return 证书的物理编号
	 */
	public static String getSignCertId() {
		try {
			Enumeration<String> aliasenum = keyStore.aliases();
			String keyAlias = null;
			if (aliasenum.hasMoreElements()) {
				keyAlias = aliasenum.nextElement();
			}
			X509Certificate cert = (X509Certificate) keyStore
					.getCertificate(keyAlias);
			return cert.getSerialNumber().toString();
		} catch (Exception e) {
			log.debug("getSignCertId Error", e);
			return null;
		}
	}


	/**
	 * 获取签名证书公钥对象
	 * 
	 * @return
	 */
	public static PublicKey getSignPublicKey() {
		try {
			Enumeration<String> aliasenum = keyStore.aliases();
			String keyAlias = null;
			if (aliasenum.hasMoreElements()) // we are readin just one
				// certificate.
			{
				keyAlias = (String) aliasenum.nextElement();
			}
			Certificate cert = keyStore.getCertificate(keyAlias);
			PublicKey pubkey = cert.getPublicKey();
			return pubkey;
		} catch (Exception e) {
			log.debug(e.toString());
			return null;
		}
	}


	/**
	 * 将证书文件读取为证书存储对象
	 * 
	 * @param pfxkeyfile
	 *            证书文件名
	 * @param keypwd
	 *            证书密码
	 * @param type
	 *            证书类型
	 * @return 证书对象
	 * @throws IOException 
	 */
	public static KeyStore getKeyInfo(String pfxkeyfile, String keypwd,
			String type) throws IOException {
		log.debug("加载签名证书==>" + pfxkeyfile);
		FileInputStream fis = null;
		try {
			KeyStore ks = null;
			if ("JKS".equals(type)) {
				ks = KeyStore.getInstance(type);
			} else if ("PKCS12".equals(type)) {
				String jdkVendor = System.getProperty("java.vm.vendor");
				String javaVersion = System.getProperty("java.version");
				log.debug("java.vm.vendor=[" + jdkVendor + "]");
				log.debug("java.version=[" + javaVersion + "]");
				//				Security.addProvider(new org.bouncycastle.jce.provider.BouncyCastleProvider());
				if (null != jdkVendor && jdkVendor.startsWith("IBM")) {
					// 如果使用IBMJDK,则强制设置BouncyCastleProvider的指定位置,解决使用IBMJDK时兼容性问题
					Security.insertProviderAt(
							new org.bouncycastle.jce.provider.BouncyCastleProvider(),
							1);
					printSysInfo();
				}else{
					Security.addProvider(new org.bouncycastle.jce.provider.BouncyCastleProvider());
				}
				//				ks = KeyStore.getInstance(type, "BC");
				ks = KeyStore.getInstance(type);
			}
			log.debug("Load RSA CertPath=[" + pfxkeyfile + "],Pwd=["
					+ keypwd + "]");
			fis = new FileInputStream(pfxkeyfile);
			char[] nPassword = null;
			nPassword = null == keypwd || "".equals(keypwd.trim()) ? null
					: keypwd.toCharArray();
			if (null != ks) {
				ks.load(fis, nPassword);
			}
			return ks;
		} catch (Exception e) {
			if (Security.getProvider("BC") == null) {
				log.debug("BC Provider not installed.");
			}
			log.debug("getKeyInfo Error", e);
			if ((e instanceof KeyStoreException) && "PKCS12".equals(type)) {
				Security.removeProvider("BC");
			}
			return null;
		}finally{
			if(null!=fis)
				fis.close();
		}
	}

	// 打印系统环境信息
	public static void printSysInfo() {
		log.debug("================= SYS INFO begin====================");
		log.debug("os_name:" + System.getProperty("os.name"));
		log.debug("os_arch:" + System.getProperty("os.arch"));
		log.debug("os_version:" + System.getProperty("os.version"));
		log.debug("java_vm_specification_version:"
				+ System.getProperty("java.vm.specification.version"));
		log.debug("java_vm_specification_vendor:"
				+ System.getProperty("java.vm.specification.vendor"));
		log.debug("java_vm_specification_name:"
				+ System.getProperty("java.vm.specification.name"));
		log.debug("java_vm_version:"
				+ System.getProperty("java.vm.version"));
		log.debug("java_vm_name:" + System.getProperty("java.vm.name"));
		log.debug("java.version:" + System.getProperty("java.version"));
		printProviders();
		log.debug("================= SYS INFO end=====================");
	}

	public static void printProviders() {
		log.debug("Providers List:");
		Provider[] providers = Security.getProviders();
		for (int i = 0; i < providers.length; i++) {
			log.debug(i + 1 + "." + providers[i].getName());
		}
	}

	/**
	 * 证书文件过滤器
	 * 
	 */
	static class CerFilter implements FilenameFilter {
		public boolean isCer(String name) {
			if (name.toLowerCase().endsWith(".cer")) {
				return true;
			} else {
				return false;
			}
		}
		public boolean accept(File dir, String name) {
			return isCer(name);
		}
	}

	/**
	 * <pre>
	 * 从一个ThreadLocal中获取当前KeyStore中的CertId,
	 * 如果获取失败则重新初始化这个KeyStore并存入ThreadLocal
	 * </pre>>
	 * @deprecated
	 * @param certPath
	 * @param certPwd
	 * @return
	 */
	public static String getCertIdByThreadLocal(String certPath, String certPwd) {
		// 初始化指定certPath和certPwd的签名证书容器
		initSignCert(certPath, certPwd);
		try {
			Enumeration<String> aliasenum = certKeyStoreLocal.get().aliases();
			String keyAlias = null;
			if (aliasenum.hasMoreElements()) {
				keyAlias = aliasenum.nextElement();
			}
			X509Certificate cert = (X509Certificate) certKeyStoreLocal.get()
					.getCertificate(keyAlias);
			return cert.getSerialNumber().toString();
		} catch (Exception e) {
			log.debug("获取签名证书的序列号失败", e);
			return "";
		}
	}

	public static String getCertIdByKeyStoreMap(String certPath, String certPwd) {
		if (!certKeyStoreMap.containsKey(certPath)) {
			// 缓存中未查询到,则加载RSA证书
			loadRsaCert(certPath, certPwd);
		}
		return getCertIdIdByStore(certKeyStoreMap.get(certPath));
	}

	private static String getCertIdIdByStore(KeyStore keyStore) {
		Enumeration<String> aliasenum = null;
		try {
			aliasenum = keyStore.aliases();
			String keyAlias = null;
			if (aliasenum.hasMoreElements()) {
				keyAlias = aliasenum.nextElement();
			}
			X509Certificate cert = (X509Certificate) keyStore
					.getCertificate(keyAlias);
			return cert.getSerialNumber().toString();
		} catch (KeyStoreException e) {
			log.debug("getCertIdIdByStore Error", e);
			return null;
		}
	}


	/**
	 * 获取证书容器
	 * 
	 * @return
	 */
	public static Map<String, X509Certificate> getCertMap() {
		return certMap;
	}

	/**
	 * 设置证书容器
	 * 
	 * @param certMap
	 */
	public static void setCertMap(Map<String, X509Certificate> certMap) {
		CertUtil.certMap = certMap;
	}

	/**
	 * 使用模和指数生成RSA公钥 注意：此代码用了默认补位方式，为RSA/None/PKCS1Padding，不同JDK默认的补位方式可能不同
	 * 
	 * @param modulus
	 *            模
	 * @param exponent
	 *            指数
	 * @return
	 */
	public static PublicKey getPublicKey(String modulus, String exponent) {
		try {
			BigInteger b1 = new BigInteger(modulus);
			BigInteger b2 = new BigInteger(exponent);
			KeyFactory keyFactory = KeyFactory.getInstance("RSA");
			RSAPublicKeySpec keySpec = new RSAPublicKeySpec(b1, b2);
			return keyFactory.generatePublic(keySpec);
		} catch (Exception e) {
			log.debug("构造RSA公钥失败：" + e);
			return null;
		}
	}
}
