package com.aote.ccb_ronglian;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.Cipher;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;

public class JsptCertUtil {

	protected static Logger logger = LoggerFactory.getLogger(JsptCertUtil.class);

	public static final String DEFAULT_CHARSET = "UTF-8";

	public static final String sign = "sign";

	public static final String EQUAL = "=";

	public static final String AMPERSAND = "&";

	/**
	 * 算法常量： SHA1
	 */
	private static final String ALGORITHM_SHA1 = "SHA-1";

	/**
	 * 算法常量：SHA1withRSA
	 */
	private static final String BC_PROV_ALGORITHM_SHA1RSA = "SHA1withRSA";

	/**
	 * 对数据通过公钥进行加密，并进行base64计算
	 * @param dataString 待处理数据
	 * @param encoding 字符编码
	 * @param key 公钥
	 * @return
	 */
	public static String encryptData(String dataString, String encoding, PublicKey key) {
		/** 使用公钥对密码加密 **/
		byte[] data = null;
		try {
			data = encryptedPin(key, dataString.getBytes(encoding));
			return new String(JsptCertUtil.base64Encode(data), encoding);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			return "";
		}
	}

	/**
	 * 通过私钥解密
	 *
	 * @param dataString base64编码过的数据
	 * @param encoding   编码
	 * @param key        私钥
	 * @return 解密后的数据
	 */
	public static String decryptedData(String dataString, String encoding, PrivateKey key) {
		try {
			byte[] data = decryptedPin(key, dataString.getBytes(encoding));
			return StringUtils.trimToEmpty(new String(data, encoding));
		} catch (Exception e) {
			logger.error("解密失败", e);
			return "";
		}
	}

	/**
	 * @param privateKey
	 * @param cryptPin
	 * @return
	 * @throws Exception
	 */
	public static byte[] decryptedPin(PrivateKey privateKey, byte[] cryptPin) throws Exception {

		// 生成PIN Block
		byte[] pinBlock = base64Decode(cryptPin);
		Cipher cipher = Cipher.getInstance("RSA/ECB/PKCS1Padding", "BC");
		cipher.init(Cipher.DECRYPT_MODE, privateKey);
		int blockSize = cipher.getBlockSize();
		int outputSize = cipher.getOutputSize(pinBlock.length);
		int leavedSize = pinBlock.length % blockSize;
		int blocksSize = leavedSize != 0 ? pinBlock.length / blockSize + 1 : pinBlock.length / blockSize;
		byte[] pinData = new byte[outputSize * blocksSize];
		int i = 0;
		while (pinBlock.length - i * blockSize > 0) {
			if (pinBlock.length - i * blockSize > blockSize) {
				cipher.doFinal(pinBlock, i * blockSize, blockSize, pinData, i * outputSize);
			} else {
				cipher.doFinal(pinBlock, i * blockSize, pinBlock.length - i * blockSize, pinData, i * outputSize);
			}
			i++;
		}
		return pinData;
	}


	/**
	 * 使用网关公钥对持卡人密码进行加密，并返回byte[]类型
	 *
	 * @param publicKey
	 * @param plainPin
	 * @return
	 * @throws Exception
	 */
	public static byte[] encryptedPin(PublicKey publicKey, byte[] plainPin) throws Exception {
		try {
			Cipher cipher = Cipher.getInstance("RSA/ECB/PKCS1Padding", "BC");
			cipher.init(Cipher.ENCRYPT_MODE, publicKey);
			int blockSize = cipher.getBlockSize();
			int outputSize = cipher.getOutputSize(plainPin.length);
			int leavedSize = plainPin.length % blockSize;
			int blocksSize = leavedSize != 0 ? plainPin.length / blockSize + 1 : plainPin.length / blockSize;
			byte[] raw = new byte[outputSize * blocksSize];
			int i = 0;
			while (plainPin.length - i * blockSize > 0) {
				if (plainPin.length - i * blockSize > blockSize) {
					cipher.doFinal(plainPin, i * blockSize, blockSize, raw, i * outputSize);
				} else {
					cipher.doFinal(plainPin, i * blockSize, plainPin.length - i * blockSize, raw, i * outputSize);
				}
				i++;
			}
			return raw;
		} catch (Exception e) {
			throw new Exception(e.getMessage());
		}
	}

	/**
	 * BASE64解码
	 * @param inputByte 待解码数据
	 * @return 解码后的数据
	 * @throws IOException
	 */
	public static byte[] base64Decode(byte[] inputByte) throws IOException {
		return Base64.decodeBase64(inputByte);
	}

	/**
	 * BASE64编码
	 * @param inputByte 待编码数据
	 * @return 解码后的数据
	 * @throws IOException
	 */
	public static byte[] base64Encode(byte[] inputByte) throws IOException {
		return Base64.encodeBase64(inputByte);
	}

	/**
	 * 生成签名值(SHA1摘要算法)
	 * @param data  待签名数据Map键值对形式
	 * @param encoding 编码
	 * @return 签名是否成功
	 */
	public static String sign(Map<String, String> data, String encoding) {
		if (StringUtils.isEmpty(encoding)) {
			encoding = DEFAULT_CHARSET;
		}
		// 将Map信息转换成key1=value1&key2=value2的形式
		String stringData = coverMap2String(data);
		logger.info("待签名请求报文串:[" + stringData + "]");
		/**
		 * 签名\base64编码
		 */
		byte[] byteSign = null;
		String stringSign = null;
		try {
			// 通过SHA1进行摘要并转16进制
			byte[] signDigest = JsptCertUtil.sha1X16(stringData, encoding);
			byteSign = JsptCertUtil.base64Encode(JsptCertUtil.signBySoft(JsptCertInit.getSignCertPrivateKey(), signDigest));
			stringSign = new String(byteSign);
			return stringSign;
		} catch (Exception e) {
			logger.error("签名异常", e);
			return stringSign;
		}
	}

	/**
	 * 将Map中的数据转换成按照Key的ascii码排序后的key1=value1&key2=value2的形式 不包含签名域signature
	 *
	 * @param data
	 *            待拼接的Map数据
	 * @return 拼接好后的字符串
	 */
	public static String coverMap2String(Map<String, String> data) {
		TreeMap<String, String> tree = new TreeMap<String, String>();
		Iterator<Entry<String, String>> it = data.entrySet().iterator();
		while (it.hasNext()) {
			Entry<String, String> en = it.next();
			if (sign.equals(en.getKey().trim())) {
				continue;
			}
			tree.put(en.getKey(), en.getValue());
		}
		it = tree.entrySet().iterator();
		StringBuffer sf = new StringBuffer();
		while (it.hasNext()) {
			Entry<String, String> en = it.next();
			sf.append(en.getKey() + EQUAL + en.getValue() + AMPERSAND);
		}
		return sf.substring(0, sf.length() - 1);
	}

	/**
	 * sha1计算后进行16进制转换
	 * @param data 待计算的数据
	 * @param encoding 编码
	 * @return 计算结果
	 */
	public static byte[] sha1X16(String data, String encoding) {
		byte[] bytes = sha1(data, encoding);
		StringBuilder sha1StrBuff = new StringBuilder();
		for (int i = 0; i < bytes.length; i++) {
			if (Integer.toHexString(0xFF & bytes[i]).length() == 1) {
				sha1StrBuff.append("0").append(Integer.toHexString(0xFF & bytes[i]));
			} else {
				sha1StrBuff.append(Integer.toHexString(0xFF & bytes[i]));
			}
		}
		try {
			return sha1StrBuff.toString().getBytes(encoding);
		} catch (UnsupportedEncodingException e) {
			logger.error(e.getMessage(), e);
			return null;
		}
	}

	/**
	 * sha1计算
	 *
	 * @param datas 待计算的数据
	 * @param encoding 字符集编码
	 * @return
	 */
	public static byte[] sha1(String datas, String encoding) {
		try {
			return sha1(datas.getBytes(encoding));
		} catch (UnsupportedEncodingException e) {
			logger.error("SHA1计算失败", e);
			return null;
		}
	}

	/**
	 * sha1计算.
	 * @param data 待计算的数据
	 * @return 计算结果
	 */
	public static byte[] sha1(byte[] data) {
		MessageDigest md = null;
		try {
			md = MessageDigest.getInstance(ALGORITHM_SHA1);
			md.reset();
			md.update(data);
			return md.digest();
		} catch (Exception e) {
			logger.error("SHA1计算失败", e);
			return null;
		}
	}

	/**
	 * 软签名
	 * @param privateKey 私钥
	 * @param data 待签名数据
	 * @return 结果
	 * @throws Exception
	 */
	public static byte[] signBySoft(PrivateKey privateKey, byte[] data) throws Exception {
		byte[] result = null;
		Signature st = Signature.getInstance(BC_PROV_ALGORITHM_SHA1RSA, "BC");
		st.initSign(privateKey);
		st.update(data);
		result = st.sign();
		return result;
	}

	/**
	 * 软验证签名
	 * @param publicKey 公钥
	 * @param signData 签名数据
	 * @param srcData 摘要
	 * @return
	 * @throws Exception
	 */
	public static boolean validateSignBySoft(PublicKey publicKey, byte[] signData, byte[] srcData) throws Exception {
		Signature st = Signature.getInstance(BC_PROV_ALGORITHM_SHA1RSA, "BC");
		st.initVerify(publicKey);
		st.update(srcData);
		return st.verify(signData);
	}
}
