package com.af.func;

import org.json.JSONArray;
import org.json.JSONObject;

import java.math.BigDecimal;
import java.util.LinkedList;
import java.util.List;
import com.af.util.TwoTuple;
import com.af.util.ByteHelper;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class Expression {
    // 加载编译好的ast文件
    public static IExpression from(InputStream ast) throws Exception {
        int type = ast.read();
        // 根据第一位标志进行处理
        switch (type) {
            // 1代表程序段
            case 1: 
                // 读取所有语句
                List<IExpression> statement = getExps(ast);
                IExpression exp = statement(statement);
                return exp;
            // 2代表函数调用
            case 2:
                // 取得函数体
                IExpression body = from(ast);
                // 取到参数列表
                statement = getExps(ast);
                return call(body, statement);
            // 3代表整数，编译结果为64位整数，运行只支持32位整数
            case 3:
                int result = ByteHelper.getInteger8(ast);
                exp = constant(result);
                return exp;
            // 4代表浮点数
            case 4:    
                double d = ByteHelper.getDouble(ast);
                exp = constant(new BigDecimal(d));
                return exp;    
            // 5代表字符串连接
            case 5:
                return concat(from(ast), from(ast));
            // 6代表字符串
            case 6:
                return constant(getString(ast));
            // 7代表数学操作
            case 7:
                String op = getString(ast);
                return math(op, from(ast), from(ast));
            // 8代表数组
            case 8:
                return array(getExps(ast));
            // 9代表map
            case 9:
                // 取长度
                int len = ByteHelper.getInteger(ast);
                List<IExpression> attrs = new LinkedList<IExpression>();
                // 找到属性值对
                for (int i = 0; i < len; i++) {
                    IExpression attr = attr(getString(ast), from(ast));
                    attrs.add(attr);
                }
                return json(attrs);
            // 10代表取属性
            case 10:
                return property(from(ast), getString(ast));
            // 11代表函数定义
            case 11:
                return func(getString(ast), getStrings(ast), from(ast));
            // 12代表取函数
            case 12:
                return funcName(getString(ast));
            // 13外部函数调用，目前不支持
            case 13:
                throw new RuntimeException("不支持外部函数调用");
            // 14代表模块引入
            case 14:
                return mod(getString(ast), from(ast));
            // 15代表比较运算符
            case 15:
                return compare(getString(ast), from(ast), from(ast));
            // 16代表Not逻辑求反
            case 16:
                return not(from(ast));
            // 17代表逻辑运算符
            case 17:
                return logic(getString(ast), from(ast), from(ast));
            // 18代表条件语句
            case 18:
                return condition(getConditions(ast), from(ast));
            // 19代表模块调用
            case 19:
                return modCall(getString(ast), getString(ast));
            default:
                throw new RuntimeException("无此类型");
        }
    }

    // 获取一批表达式，前4个字节表示表达式个数
    public static List<IExpression> getExps(InputStream ast) throws Exception {
        // 获取32位长度
        int len = ByteHelper.getInteger(ast);
        // 读取所有语句
        List<IExpression> statement = new LinkedList<IExpression>();
        for (int i = 0; i < len; i++) {
            IExpression exp = from(ast);
            statement.add(exp);
        }
        return statement;
    }

    // 获取一批字符串，前4个字节表示个数
    public static List<Object> getStrings(InputStream ast) throws Exception {
        // 获取32位长度
        int len = ByteHelper.getInteger(ast);
        // 读取所有字符串
        List<Object> statement = new LinkedList<Object>();
        for (int i = 0; i < len; i++) {
            Object str = getString(ast);
            statement.add(str);
        }
        return statement;
    }

    // 获取一批条件，前4个字节表示个数
    public static List<TwoTuple<IExpression, IExpression>> getConditions(InputStream ast) throws Exception {
        // 获取32位长度
        int len = ByteHelper.getInteger(ast);
        // 读取所有语句
        List<TwoTuple<IExpression, IExpression>> statement = new LinkedList<>();
        for (int i = 0; i < len; i++) {
            statement.add(new TwoTuple(from(ast), from(ast)));
        }
        return statement;
    }

    // 取字符串，前四个字节代表字符串字节数
    public static String getString(InputStream ast) throws Exception {
        // 取长度
        int len = ByteHelper.getInteger(ast);
        // 读取给定长度字节数
        byte[] bytes = new byte[len];
        ast.read(bytes);
        // 把字节数组转化成字符串
        return new String(bytes);
    }

    // 数学运算
    public static IExpression math(String op, IExpression left, IExpression right) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                Object oLeft = left.invoke(del);
                Object oRight = right.invoke(del);

                // 只要有BigDecimal，按BigDecimal计算，否则，按int计算
                if (oLeft instanceof BigDecimal || oRight instanceof BigDecimal) {
                    BigDecimal l = new BigDecimal(oLeft.toString());
                    BigDecimal r = new BigDecimal(oRight.toString());
                    switch (op) {
                        case "+":
                            return l.add(r);
                        case "-":
                            return l.subtract(r);
                        case "*":
                            return l.multiply(r);
                        case "/":
                            return l.divide(r);
                        case "%":
                            return new BigDecimal(l.intValue() % r.intValue());
                        default:
                            throw new RuntimeException("不识别的算数操作符");
                    }
                } else {
                    int l = Integer.parseInt(oLeft.toString());
                    int r = Integer.parseInt(oRight.toString());
                    switch (op) {
                        case "+":
                            return l + r;
                        case "-":
                            return l - r;
                        case "*":
                            return l * r;
                        case "/":
                            return l / r;
                        case "%":
                            return l % r;
                        default:
                            throw new RuntimeException("不识别的算数操作符");
                    }
                }
            }

            public IExpression partialCall(Delegate del, List<String> params) {
                return math(op, left.partialCall(del, params), right.partialCall(del, params));
            }

            public String toString() {
                return "Math(" + op + ", " + left.toString() + ", " + right.toString() + ")";
            }
        };
    }

    // 比较运算
    public static IExpression compare(String op, IExpression left, IExpression right) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                Object l = left.invoke(del);
                Object r = right.invoke(del);

                // 如果有一个是bigdecimal，全部按bigdecimal处理
                if (l instanceof BigDecimal || r instanceof BigDecimal) {
                    l = new BigDecimal(l.toString());
                    r = new BigDecimal(r.toString());
                }

                // 可以进行大小比较，取比较大小的结果
                if (l instanceof Comparable && r instanceof Comparable) {
                    int result = ((Comparable) l).compareTo(r);
                    switch (op) {
                        case ">":
                            return result > 0;
                        case ">=":
                            return result >= 0;
                        case "<":
                            return result < 0;
                        case "<=":
                            return result <= 0;
                        case "==":
                            return result == 0;
                        case "!=":
                            return result != 0;
                    }
                } else {
                    // 只进行相等比较
                    boolean equals = l.equals(r);
                    switch (op) {
                        case "==":
                            return equals;
                        case "!=":
                            return !equals;
                    }
                }

                return null;
            }

            public IExpression partialCall(Delegate del, List<String> params) {
                return compare(op, left.partialCall(del, params), right.partialCall(del, params));
            }

            public String toString() {
                return "Compare(" + op + ", " + left.toString() + ", " + right.toString() + ")";
            }
        };
    }

    // 逻辑运算
    public static IExpression logic(String op, IExpression left, IExpression right) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                boolean l = (boolean)left.invoke(del);
                boolean r = (boolean)right.invoke(del);
                switch (op) {
                    case "||":
                        return l || r;
                    case "&&":
                        return l && r;
                    default:
                        throw new RuntimeException("未知类型");
                }
            }

            public IExpression partialCall(Delegate del, List<String> params) {
                return logic(op, left.partialCall(del, params), right.partialCall(del, params));
            }

            public String toString() {
                return "Logic(" + op + ", " + left.toString() + ", " + right.toString() + ")";
            }
        };
    }

    // 逻辑非运算
    public static IExpression not(IExpression exp) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                return !(boolean)exp.invoke(del);
            }

            public IExpression partialCall(Delegate del, List<String> params) {
                return not(exp.partialCall(del, params));
            }

            public String toString() {
                return "Not(" + exp.toString() + ")";
            }
        };
    }

    // 产生json对象
    public static IExpression json(List<IExpression> children) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                JSONObject result = new JSONObject();
                for (IExpression child : children) {
                    Pair pair = (Pair)child.invoke(del);
                    result.put(pair.name, pair.value);
                }
                return result;
            }

            public IExpression partialCall(Delegate del, List<String> params) {
                // 把每一个子进行调用
                List<IExpression> result = children.stream().map(s -> s.partialCall(del, params)).collect(Collectors.toList());
                return json(result);
            }

            public String toString() {
                return "Json(" + children.toString() + ")";
            }
        };
    }

    // 属性值对
    static private class Pair {
        String name;
        Object value;
        public Pair(String name, Object value) {
            this.name = name;
            this.value = value;
        }
    }

    // 产生属性值对
    public static IExpression attr(String name, IExpression v) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                return new Pair(name, v.invoke(del));
            }

            public IExpression partialCall(Delegate del, List<String> params) {
                return attr(name, v.partialCall(del, params));
            }

            public String toString() {
                return "Attr(" + name + ", " + v.toString() + ")";
            }
        };
    }

    // 产生JSON数组
    public static IExpression array(List<IExpression> children) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                JSONArray result = new JSONArray();
                for (IExpression child : children) {
                    result.put(child.invoke(del));
                }
                return result;
            }

            public IExpression partialCall(Delegate del, List<String> params) {
                // 把每一个子进行调用
                List<IExpression> result = children.stream().map(s -> s.partialCall(del, params)).collect(Collectors.toList());
                return array(result);
            }

            public String toString() {
                return "Array(" + children.toString() + ")";
            }
        };
    }

    // 产生元组，拿数组表示元组
    public static IExpression tuple(List<IExpression> children) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                Object[] result = new Object[children.size()];
                int i = 0;
                for (IExpression child : children) {
                    result[i] = child.invoke(del);
                    i++;
                }
                return result;
            }

            public IExpression partialCall(Delegate del, List<String> params) {
                // 把每一个子进行调用
                List<IExpression> result = children.stream().map(s -> s.partialCall(del, params)).collect(Collectors.toList());
                return tuple(result);
            }

            public String toString() {
                return "Tuple(" + children.toString() + ")";
            }
        };
    }

    // 元组解构，把元组解构放到具体值中
    public static IExpression deattach(List<String> params, IExpression exp) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                // 把执行结果放到变量代表的无参函数中
                Object[] result = (Object[])exp.invoke(del);
                int i = 0;
                for (String name : params) {
                    del.putFunc(name, new LinkedList<>(), constant(result[i]));
                    i++;
                }
                return null;
            }

            public IExpression partialCall(Delegate del, List<String> invokeParams) {
                return deattach(params, exp.partialCall(del, invokeParams));
            }

            public String toString() {
                return "Deattach(" + params.toString() + ", " + exp.toString() + ")";
            }
       };
    }

    // 取属性值
    public static IExpression property(IExpression objExp, String name) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                // 获取对象
                Object obj = objExp.invoke(del);

                // 是JSONObject
                if (obj instanceof JSONObject) {
                    JSONObject json = (JSONObject) obj;
                    return json.get(name);
                } else {
                    // 是插件，返回插件中的函数，插件只支持函数
                    return new Delegate.JavaFunc(obj, name);
                }
            }

            public IExpression partialCall(Delegate del, List<String> params) {
                return property(objExp.partialCall(del, params), name);
            }

            public String toString() {
                return "Property(" + objExp.toString() + ", " + name + ")";
            }
        };
    }

    // 产生常数
    public static IExpression constant(Object value) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                return value;
            }

            public IExpression partialCall(Delegate del, List<String> params) {
                return constant(value);
            }

            public String toString() {
                return "Constant(" + value.toString() + ")";
            }
        };
    }

    // 产生函数，把产生的函数保存到函数表中，方便后续调用
    public static IExpression func(String name, List<Object> params, IExpression exp) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                // 把函数定义追加到上下文环境中
                del.putFunc(name, params, exp);
                return null;
            }

            public IExpression partialCall(Delegate del, List<String> invokeParams) {
                return func(name, params, exp.partialCall(del, invokeParams));
            }

            public String toString() {
                return "Func(" + name + ", " + params.toString() + ", " + exp.toString() + ")";
            }
        };
    }

    // 产生模块引入
    public static IExpression mod(String name, IExpression exp) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                // 把模块添加到上下文环境中
                del.mods.put(name, exp);
                return null;
            }

            public IExpression partialCall(Delegate del, List<String> invokeParams) {
                return mod(name, exp.partialCall(del, invokeParams));
            }

            public String toString() {
                return "mod(" + name + ", " + exp.toString() + ")";
            }
        };
    }

    // 产生模块调用
    public static IExpression modCall(String modName, String funcName) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                // 返回模块调用函数
                return new Delegate.ModFunc(modName, funcName);
            }

            public IExpression partialCall(Delegate del, List<String> invokeParams) {
                return modCall(modName, funcName);
            }

            public String toString() {
                return "modCall(" + modName + ", " + funcName + ")";
            }
       };
    }

    // 根据函数名，获取函数
    public static IExpression funcName(String name) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                // 先从插件里找java对象
                if (del.plugins.containsKey(name)) {
                    return del.plugins.get(name);
                } else if (del.funcs.containsKey(name)) {
                    Delegate.Func result = del.getFunc(name);
                    // 获取给定函数                
                    return result;
                } else {
                    // 函数不存在，报错
                    throw new RuntimeException("函数不存在：" + name);
                }
            }

            public IExpression partialCall(Delegate del, List<String> names) {
                // 如果是已经提供的参数，拿参数执行结果替换，否则，复制
                if (names.contains(name)) {
                    // 从上下文中取出参数内容执行
                    Object result = del.getFunc(name).body.invoke(del);
                    // 把执行结果转换成常数表达式
                    return constant(result);
                } else {
                    return funcName(name);
                }
            }

            public String toString() {
                return "funcName(" + name + ")";
            }
        };
    }

    // 执行条件运算
    public static IExpression condition(List<TwoTuple<IExpression, IExpression>> conds, IExpression elseExp) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                // 有一个满足，返回满足条件的结果
                for (TwoTuple<IExpression, IExpression> item : conds) {
                    if ((boolean)item.first.invoke(del)) {
                        return item.second.invoke(del);
                    }
                }
                // 都不满足，返回else结果
                return elseExp.invoke(del);
            }

            public IExpression partialCall(Delegate del, List<String> names) {
                // 把每一个子进行调用
                List<TwoTuple<IExpression, IExpression>> result = new LinkedList<>();
                for (TwoTuple<IExpression, IExpression> s : conds) {
                    result.add(new TwoTuple(s.first.partialCall(del, names), s.second.partialCall(del, names)));
                }

                return condition(result, elseExp.partialCall(del, names));
            }

            public String toString() {
                return "condition(" + conds.toString() + ", " + elseExp.toString() + ")";
            }
       };
    }

    // 执行字符串连接
    public static IExpression concat(IExpression left, IExpression right) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                return left.invoke(del).toString() + right.invoke(del).toString();
            }

            public IExpression partialCall(Delegate del, List<String> names) {
                return concat(left.partialCall(del, names), right.partialCall(del, names));
            }

            public String toString() {
                return "concat(" + left.toString() + ", " + right.toString() + ")";
            }
        };
    }

    // 执行函数调用，不是java对象函数调用，是调用脚本里定义的函数
    public static IExpression call(IExpression body, List<IExpression> params) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                // 执行body部分
                Object obj = body.invoke(del);
                // 如果执行结果是函数，继续进行函数调用
                if (obj instanceof Delegate.Func) {
                    return loopCall((Delegate.Func)obj, params, del, 0);
                } else if (obj instanceof Delegate.ModFunc) {
                    // 模块调用，调用模块中的函数
                    Delegate.ModFunc func = (Delegate.ModFunc)obj;
                    return modCall(func.modName, func.funcName, params, del);
                } else if (obj instanceof Delegate.JavaFunc) {
                    // java插件中的函数，进行java插件中的函数调用
                    Delegate.JavaFunc func = (Delegate.JavaFunc)obj;
                    try {
                        return javaCall(func.obj, func.funcName, params, del);
                    } catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                } else {
                    // 返回执行结果
                    return obj;
                }
            }

            public IExpression partialCall(Delegate del, List<String> names) {
                // 把每一个子进行调用
                List<IExpression> result = params.stream().map(
                    s -> s.partialCall(del, names)
                ).collect(Collectors.toList());

                return call(body.partialCall(del, names), result);
            }

            public String toString() {
                return "call(" + body.toString() + ", " + params.toString() + ")";
            }
        };
    }

    // 产生语句处理过程
    public static IExpression statement(List<IExpression> states) {
        return new IExpression() {
            public Object invoke(Delegate del) {
                Object result = null;
                // 所有语句循环执行，返回最后一个语句的执行结果
                for (IExpression child : states) {
                    result = child.invoke(del);
                }
                return result;
            }

            public IExpression partialCall(Delegate del, List<String> names) {
                // 把每一个子进行调用
                List<IExpression> result = states.stream().map(
                    s -> s.partialCall(del, names)
                ).collect(Collectors.toList());

                return statement(result);
            }

            public String toString() {
                return "statement(" + states.toString() + ")";
            }
        };
    }

    // 调用java插件中的函数
    public static Object javaCall(Object obj, String funcName, List<IExpression> params, Delegate del) throws Exception {
        // 对参数值进行计算，以方便传递给java函数
        List<Object> realParams = params.stream().map(exp -> exp.invoke(del)).collect(Collectors.toList());
        // 取得参数对应的类型，以便获取正确函数
        Class[] paramTypes = realParams.stream().map(p -> p.getClass()).toArray(Class[]::new);
        // 利用反射取得函数
        Method method = obj.getClass().getMethod(funcName, paramTypes);
        return method.invoke(obj, realParams.toArray());
    }

    // 调用模块中的函数
    public static Object modCall(String modName, String funcName, List<IExpression> params, Delegate del) {
        // 找到模块对应的程序
        if (del.mods.containsKey(modName)) {
            IExpression exp = del.mods.get(modName);
            // 执行表达式，获得新上下文
            Delegate newDel = new Delegate();
            exp.invoke(newDel);
            // 从上下文中找到函数，进行调用，调用时，用模块中的上下文
            if (newDel.funcs.containsKey(funcName)) {
                return loopCall(newDel.getFunc(funcName), params, newDel, 0);
            } else {
                throw new RuntimeException("函数不存在：" + modName + ":" + funcName);
            }
        } else {
            throw new RuntimeException("模块不存在：" + modName + ":" + funcName);
        }
    }

    // 对函数循环进行调用，直到函数执行结果不是函数为止
    public static Object loopCall(Delegate.Func func, List<IExpression> params, Delegate del, int start) {
        // 执行以便调用
        Object obj = oneCall(func, params, del, start);
        // 如果执行结果是函数，拿剩余参数继续调用
        if (obj instanceof Delegate.Func) {
            Delegate.Func newFunc = (Delegate.Func)obj;
            // 如果函数需要参数，但没有参数，直接返回这个函数，否则，继续调用
            if (newFunc.params.size() == 0 || params.size() > start + func.params.size()) {
                int newStart = start + func.params.size();
                return loopCall((Delegate.Func)obj, params, del, newStart);
            } else {
                return obj;
            }
        } else {
            // 返回调用结果
            return obj;
        }
    }

    // 执行一遍实际函数调用
    // @param start: 该函数从start处开始取参数
    public static Object oneCall(Delegate.Func func, List<IExpression> params, Delegate del, int start) {
        // 用复制出来的上下文进行函数调用，以便保护函数调用现场
        Delegate newDel = del.clone();
        // 把参数放到上下文环境中
        int len = func.params.size();
        int i = 0;
        while (i + start < params.size()) {
            // 如果实参超过虚参个数，剩余参数不是这个函数调用自身的，留下来给后面的函数调用
            if (i >= len) {
                break;
            }
            putParam(func.params.get(i), params.get(i + start), newDel);
            i += 1;
        }
        // 执行函数调用过程，有可能是部分调用
        Object result = funcCall(func, newDel, i);
        return result;
    }

    // 执行函数调用，调用结果可能是部分函数
    // @param num: 剩余实参个数
    // @param params: 所有实参
    private static Object funcCall(Delegate.Func func, Delegate del, int num) {
        // 实参个数够，直接调用
        if (func.params.size() == num) {
            return func.body.invoke(del);
        } else {
            // 把前面的参数名，转换成字符串列表
            List<String> names = func.params.stream().limit(num).map(s -> s.toString()).collect(Collectors.toList());
            // 返回部分调用的结果
            return new Delegate.Func(
                // 函数虚参为截取后面的参数结果
                func.params.stream().skip(num).collect(Collectors.toList()),
                // 函数体为部分调用的结果，参数为前面已经提供的参数
                func.body.partialCall(del, names)
            );
        }
    }

    // 把函数调用的参数放到执行环境中，函数参数支持元组等的解构过程
    private static void putParam(Object param, IExpression exp, Delegate del) {
        // 字符串，不用解构，直接放进去
        if (param instanceof String) {
            // 实参要执行，实参执行结果转换成常数表达式
            IExpression e = constant(exp.invoke(del));
            // 把实参执行结果放到执行环境中
            del.putFunc((String)param, new LinkedList<>(), e);
        } else {
            // 按元组进行解构，元组的内容可能还是元组，需要递归调用
            Object[] values = (Object[])exp.invoke(del);
            int i = 0;
            for (Object p : (List)param) {
                putParam(p, constant(values[i]), del);
                i++;
            }
        }
    }
}
