package com.af.expression;

import com.af.expression.exception.ReturnWayException;

import java.util.HashMap;
import java.util.Map;

/**
 * 执行单元，保存所有虚参，Expression编译后的结果
 *
 * @author 何宁社
 */
public class Delegate {
	/**
	 * 执行时的Expression
	 */
	private final Expression exp;
	/**
	 * 执行的源代码
	 */
	private final String source;
	/**
	 * 参数值及变量值存放处
	 */
	private Map<String, Object> objectNames = new HashMap<>();

	/**
	 * 用Expression构造，执行时最顶层Expression
	 * @param exp 表达式
	 */
	public Delegate(Expression exp, String source) {
		this.exp = exp;
		this.source = source;
	}

	/**
	 * 无参执行过程
	 * @return 结果
	 */
	public Object invoke() {
		return this.invoke(new HashMap<>(8));
	}

	/**
	 * 执行程序，执行前，参数必须实例化
	 * @param params 执行时，所带参数值
	 * @return 执行结果
	 */
	public Object invoke(Map<String, Object> params) {
		// 把初始参数给参数表
		setObjectNames(params);
		// 沿根Expression节点遍历，把delegate传递下去
		putDelegate(this.exp);
		// 调用exp的执行过程
		try {
			return exp.invoke();
		} catch (ReturnWayException returnWay) {
			return returnWay.getReturnObject();
		}
	}

	/**
	 * 沿根节点递归，传递delegate的过程
	 * @param parent 表达式
	 */
	private void putDelegate(Expression parent) {
		for (Expression child : parent.children) {
			// 有些节点会放空的子节点
			if (child == null) {
				continue;
			}
			//try特殊处理
			if ("Try".equals(child.type.name())) {
				Expression[] exps = (Expression[]) child.value;
				if (exps[0] != null) {
					exps[0].setDelegate(this);
					putDelegate(exps[0]);
				}
				if (exps[1] != null) {
					exps[1].setDelegate(this);
					putDelegate(exps[1]);
				}
			}
			child.setDelegate(this);
			putDelegate(child);
		}
	}

	public Expression getExp() {
		return this.exp;
	}

	public void setObjectNames(Map<String, Object> objectNames){
		this.objectNames = objectNames;
	}

	public Map<String, Object> getObjectNames() {
		return objectNames;
	}

	public void put(String key, Object value){
		objectNames.put(key, value);
	}

	public Object get(String key){
		return objectNames.get(key);
	}

	public boolean containsKey(String key){
		return objectNames.containsKey(key);
	}

	public String getSource() {
		return source;
	}
}
