package com.af.v4.system.runtime.service;

import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.service.ApplicationService;
import com.af.v4.system.common.expression.core.Program;
import com.af.v4.system.common.plugins.core.CommonTools;
import com.af.v4.system.common.resource.mapper.AbstractResourceMapper;
import jakarta.annotation.PostConstruct;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 业务资源检测服务
 *
 * @author Mr.river
 */
@Service
public class ResCheckService {

    private static final Logger LOGGER = LoggerFactory.getLogger(ResCheckService.class);

    private static final List<String> ERROR_ARRAY = new ArrayList<>();
    private static final String[] sentence = {
            "软件开发往往是这样：最开始的90%代码占用了开始的90%的开发时间；剩下10%代码同样需要90%的开发时间。",
            "这不是一个bug，这只是一个未列出来的特性。",
            "作为一个程序员，郁闷的事情是，⾯对一个代码块，却不敢去修改。更糟糕的是，这个代码块还是⾃⼰写的。",
            "在系统测试阶段查找并修复 Bug，花费的时间和工作量是让开发者自己找 Bug 的三倍。在正式上线后查找并修复 Bug，花费的时间和工作量是系统测试阶段的十倍。因此一定要让开发者自己做单元测试。",
            "不要站着调试。站着会让你的耐心减半，而且你需要集中所有精力。",
            "测试可以发现bug的存在，但不能证明bug不存在。",
            "新系统的每个新用户都会发现一类新的Bug。",
            "修复损坏的程序的第一步就是让它不断失败。",
            "有两种方法能写出没有错误的程序；但只有第三种好用。",
            "最危险的是：你写的代码六个月后自己都看不懂。",
            "程序员的三大谎言：这是个简单的改动；这个bug我修好了；代码写完了。",
            "写代码要像写情书一样优雅，要像写遗嘱一样严谨。",
            "代码写的越急，程序跑得越慢。"
    };
    private static final String[] prompt = {
            "我们有过各种创伤，但我们今天应该快活。",
            "四处走走，你会热爱这个世界。",
            "一定要爱着点什么，它让我们变得坚韧、宽容、充盈。业余的，爱着。",
            "一天晚上，铺子都关了，街上已断行人，路灯照着空荡荡的马路，而远远的一个理发店标记在冷静之中孤零零地动。这一下子把你跟世界拉得很近，犹如大漠孤烟。",
            "活着，就还得做一点事。 我们有过各种创伤，但我们今天应该快活。",
            "哪里有鲜花，就到哪里去。",
            "走在路上，间或也能见一个钉碗的，之故之故拉他的金刚钻；一个补锅的，用一个布卷在灰上一揉，托起一小勺殷红的熔铁，嗤地一声焊在一口三眼灶大黑锅上；一个皮匠，把刀在他的脑后发桩子上光一光，这可以让你看半天。 你看他们工作，也看他们人。他们是一种“遗民”，永远固执而沉默地慢慢地走，让你觉得许多事情值得深思。",
            "我不是不食人间烟火，不懂感情的人。我不喜欢那种口不臧否人物，绝不议论朝政，无爱无憎，无是无非，胆小怕事，除了猪肉白菜的价格什么也不关心的离退休干部。这种人有的是。 中国人有一种哲学，叫做“忍”。我小时候听过“百忍堂”张家的故事，就非常讨厌。现在一些名胜古迹卖碑帖的文物商店卖的书法拓本最多的一是郑板桥的“难得糊涂”，二是一个大字：“忍”。这是一种非常庸俗的人生哲学。",
            "人不管走到哪一步，总得找点乐子，想一点办法，老是愁眉苦脸的，干吗呢？",
            "写字作画，首先得有激情。要有情绪，为一人、一事、一朵花、一片色彩感动。有一种意向、一团兴致，勃勃然郁积于胸，势欲喷吐而出。",
            "我在香港时全像一根落在泥水里的鸡毛，没有话说，我沾湿了，弄脏了，不成样子。忧郁，一种毫无意义的忧郁。我一定非常丑，我脸上线条零乱芜杂，我动作萎靡鄙陋，我不跟人说话，我若一开口一定不知所云！我真不知道我怎么把自己糟蹋到这种地步。",
            "写字作画,首先得有激情。要有情绪,为人、一事、一朵花、一片色彩感动。有种意向、一团兴致,勃勃然郁积于胸,势欲喷吐而出。爱,是一件非专业的事情,不是本事,不是能力,是花木那样的生长,有一份对光阴和季节的钟情和执着。一定要,爱着点什么,它让我们变得坚韧,宽容,充盈。业余着，爱着。",
            "鸟为什么要“遛”？不遛不叫。鸟必须习惯于笼养，习惯于喧闹扰攘的环境。等到它习惯于与人相处时，它就会尽情鸣叫。",
            "记得旧时好，跟随爹爹去吃茶。门前磨螺壳，巷口弄泥沙。",
            "在人生的每个阶段，我们都对漫漫前程抱着一份激动的希望，以为奇迹就在前方。然而，人生只是一个个梦想不断破灭的过程。而当我们走出所有曲折的日子时才发现，真正的美好与神奇，已经永远地留在了背后。",
            "但是太阳，他每时每刻都是夕阳也都是旭日。当他熄灭着走下山去收尽苍凉残照之际，正是他在另一面燃烧着爬上山巅布散烈烈朝辉之时。",
            "那一天，我也将沉静着走下山去，扶着我的拐杖。有一天，在某一处山洼里，势必会跑上来一个欢蹦的孩子，抱着他的玩具。当然，那不是我。但是，那不是我吗？",
            "那一天我二十一岁，在我一生的黄金时代，我有好多奢望。我想爱，想吃，还想在一瞬间变成天上半明半暗的云。",
            "生活就是个缓慢受锤的过程，人一天天老下去，奢望也一天天消失，最后变得像挨了锤的牛一样。",
            "围在城里的人想逃出来，城外的人想冲进去。对于婚姻也罢，职业也罢，人生的愿望大都如此。",
            "每个人都有属于自己的一片森林，迷失的人迷失了，相逢的人会再相逢。",
            "孤独一人也没关系，只要能发自内心地爱着一个人，人生就会有救。",
            "所谓活着并不是单纯的呼吸，心脏跳动，而是在这个世界上留下痕迹。",
            "世界上没有一条道路是重复的，也没有一个人生是可以替代的。",
            "卑鄙是卑鄙者的通行证，高尚是高尚者的墓志铭。",
            "乡愁是一枚小小的邮票，我在这头，母亲在那头。",
            "从前的日色变得慢，车，马，邮件都慢，一生只够爱一个人。",
            "生活不止眼前的苟且，还有诗和远方的田野。",
            "如果有来生，要做一棵树，站成永恒，没有悲欢的姿势。",
            "岁月极美，在于它必然流逝。春花，秋月，夏日，冬雪。",
            "人活一世，草木一秋，无论长短，总得有些意思。",
            "不要因为走得太远，而忘记为什么出发。",
            "人生如逆旅，我亦是行人。",
            "你知道的越多，你不知道的越多。",
            "知我者谓我心忧，不知我者谓我何求。",
            "生活是很好玩的，它会带来很多惊喜，也会有些失落，但无论如何，都要走下去。",
            "燕子去了，有再来的时候；杨柳枯了，有再青的时候；桃花谢了，有再开的时候，但是聪明的，你告诉我，我们的日子为什么一去不复返呢？"
    };

    private final ApplicationService applicationService;

    private final List<AbstractResourceMapper<?>> abstractResourceMappers;

    public ResCheckService(ApplicationService applicationService, List<AbstractResourceMapper<?>> abstractResourceMappers) {
        this.applicationService = applicationService;
        this.abstractResourceMappers = abstractResourceMappers;
    }

    @PostConstruct
    public void checkAll() {
        EnvType envType = applicationService.getEnvType();
        if (envType == EnvType.PROD) {
            LOGGER.info("==================跳过资源检查==================");
            return;
        }
        LOGGER.info("==================开始资源强制检查==================");
        boolean throwError = false;
        StringBuilder msg = new StringBuilder("检查结果：");
        for (AbstractResourceMapper<?> mapper : abstractResourceMappers) {
            int errorNum = checkRes(mapper);
            if (errorNum > 0) {
                throwError = true;
                msg.append("\n * >>>ERROR: ").append(mapper.getResType())
                        .append("强制检查未通过，共有").append(errorNum).append("项错误");
            }
        }
        if (throwError) {
            String errorWelcome = """
                    /***
                     * _ooOoo_
                     * o8888888o
                     * 88" . "88
                     * (| -_- |)
                     *  O\\ = /O
                     * ___/`---'\\____
                     * .   ' \\\\| |// `.
                     * / \\\\||| : |||// \\
                     * / _||||| -:- |||||- \\
                     * | | \\\\\\ - /// | |
                     * | \\_| ''\\---/'' | |
                     * \\ .-\\__ `-` ___/-. /
                     * ___`. .' /--.--\\ `. . __
                     * ."" '< `.___\\_<|>_/___.' >'"".
                     * | | : `- \\`.;`\\ _ /`;.`/ - ` : | |
                     * \\ \\ `-. \\_ __\\ /__ _/ .-` / /
                     * ======`-.____`-.___\\_____/___.-`____.-'======
                     * `=---='
                     * .............................................
                     * 佛曰：bug泛滥，我已瘫痪！
                    """;
            msg.append("\n * >>> 至理名言：").append(sentence[CommonTools.getRandomNumber(0, sentence.length - 1)]);
            errorWelcome = errorWelcome + "\n * " + msg + "\n";
            LOGGER.error(errorWelcome);
            for (int i = 0; i < ERROR_ARRAY.size(); i++) {
                StringBuilder item = new StringBuilder();
                LOGGER.error(item.append(" * ").append(i + 1).append(". ").append(ERROR_ARRAY.get(i)).toString());
            }
            LOGGER.error("==================资源强制检查完成==================\n");
            try {
                //此处睡眠是为了避免异步日志未输出完就抛出异常
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
            throw new ServiceException("资源检查未通过，程序已强制中止");
        } else {
            String successWelcome = """
                    /***
                     *                    _ooOoo_
                     *                   o8888888o
                     *                   88" . "88
                     *                   (| -_- |)
                     *                    O\\ = /O
                     *                ____/`---'\\____
                     *              .   ' \\\\| |// `.
                     *               / \\\\||| : |||// \\
                     *             / _||||| -:- |||||- \\
                     *               | | \\\\\\ - /// | |
                     *             | \\_| ''\\---/'' | |
                     *              \\ .-\\__ `-` ___/-. /
                     *           ___`. .' /--.--\\ `. . __
                     *        ."" '< `.___\\_<|>_/___.' >'"".
                     *       | | : `- \\`.;`\\ _ /`;.`/ - ` : | |
                     *         \\ \\ `-. \\_ __\\ /__ _/ .-` / /
                     * ======`-.____`-.___\\_____/___.-`____.-'======
                     *                    `=---='
                     *
                     * .............................................
                     *          佛祖保佑             永无BUG
                     */""";
            LOGGER.info(successWelcome);
            LOGGER.info("送给你的：{}", prompt[CommonTools.getRandomNumber(0, prompt.length - 1)]);
        }
    }

    /**
     * 资源检测
     */
    private int checkRes(AbstractResourceMapper<? extends AbstractResourceMapper.CommonResource> mapper) {
        Map<String, ?> resMap = mapper.getAllMap();
        AtomicInteger i = new AtomicInteger(1);
        AtomicInteger success = new AtomicInteger(0);
        AtomicInteger error = new AtomicInteger(0);
        int size = resMap.size();
        LOGGER.info("检查{}资源，共 {} 个...", mapper.getResType(), size);
        resMap.forEach((key, value) -> {
            // 获取源程序内容
            AbstractResourceMapper.CommonResource resource = mapper.getResource(key, true, false);
            String source;
            assert resource != null;
            String alias = resource.getAlias();
            String path = resource.getPath();
            try {
                if (mapper.isSupportCompile()) {
                    source = resource.getSource();
                    Program pros = new Program(source);
                    // 解析
                    try {
                        pros.parse();
                        success.addAndGet(1);
                    } catch (RuntimeException e) {
                        String message = e.getMessage();
                        ERROR_ARRAY.add(mapper.getResType() + "资源[" + alias + "](" + path + ")编译错误，" + message);
                        error.addAndGet(1);
                    }
                } else {
                    success.addAndGet(1);
                }
            } catch (RuntimeException e) {
                ERROR_ARRAY.add(mapper.getResType() + "资源文件[" + alias + "]指向了一个不存在的位置：" + path);
                error.addAndGet(1);
            }
            i.addAndGet(1);
        });
        String resultMsg = mapper.getResType() + "资源检查完成，共 " + size + " 个,正确 " + success.get() + " 个,错误 " + error.get() + " 个";
        if (error.get() > 0) {
            LOGGER.error(resultMsg);
        } else {
            LOGGER.info(resultMsg);
        }
        return error.get();
    }
}
