package com.af.v4.system.runtime.controller;

import com.af.v4.system.common.core.constant.ServiceNameConstants;
import com.af.v4.system.common.core.web.domain.AjaxResult;
import com.af.v4.system.common.core.utils.ServletUtils;
import com.af.v4.system.common.log.annotation.Log;
import com.af.v4.system.common.log.enums.BusinessType;
import com.af.v4.system.common.logic.service.LogicService;
import com.af.v4.system.common.security.utils.SecurityUtils;
import com.af.v4.system.common.plugins.security.SubstitutionCipherUtil;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.*;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;

/**
 * 脱敏数据查询控制器
 * 提供对加密存储的敏感数据进行安全查询的接口
 */
@RestController
@RequestMapping("/sensitive")
public class SensitiveDataController {

    @Autowired
    private LogicService logicService;


    /**
     * 对加密数据进行解密（仅管理员可用）
     * 用于在需要查看原始数据时进行解密操作，需要管理员权限
     *
     * @param request 密文数据
     * @return 解密后的明文
     */
    @PostMapping("/decrypt")
    @Log(title = "敏感数据解密", businessType = BusinessType.OTHER)
    public AjaxResult decrypt(@RequestBody Map<String, String> request) {
        try {
            // 检查是否为管理员
            if (!SecurityUtils.isAdmin(SecurityUtils.getUserId())) {
                return AjaxResult.error(HttpStatus.FORBIDDEN.value(), "无权限执行解密操作");
            }

            String cipherText = request.get("cipherText");
            if (cipherText == null || cipherText.isEmpty()) {
                return AjaxResult.error("密文数据不能为空");
            }

            String plainText = SubstitutionCipherUtil.decrypt(cipherText);
            Map<String, String> result = new HashMap<>();
            result.put("plainText", plainText);

            // 记录解密操作日志（注意：只记录密文，不记录明文）
            logOperation(cipherText, "t_user_sensitive_crypto", "解密字段");

            return AjaxResult.success(result);
        } catch (Exception e) {
            String cipherText = request != null ? request.get("cipherText") : null;
            logOperation( cipherText, "t_user_sensitive_crypto", "解密字段");
            return AjaxResult.error("解密失败: " + e.getMessage());
        }
    }

    /**
     * 查询包含敏感数据的记录
     * 通过加密后的关键字进行查询，避免明文传输和存储
     *
     * @param data 查询参数，包含加密的关键字和查询条件
     * @return 查询结果（包含解密后的明文数据）
     */
    @PostMapping("/query")
    @Log(title = "敏感数据查询", businessType = BusinessType.OTHER)
    public AjaxResult query(@RequestBody Map<String, Object> data) {
        Map<String, Object> params = null;
        try {
            // 验证必要参数
            if (data == null || data.isEmpty()) {
                return AjaxResult.error("查询参数不能为空");
            }
            params = (Map<String, Object>) data;

            // 获取查询条件中的密文字段（如果有）
            String cipherData = params.get("cipherText") != null ? params.get("cipherText").toString() : null;
            String operateTable = params.get("tableName") != null ? params.get("tableName").toString() : "";
            String operateField = params.get("field") != null ? params.get("field").toString() : "敏感字段";
            Map<String, String> dataMap = new HashMap<>();
            dataMap.put(operateField, cipherData);
            decryptSensitiveFields(dataMap);
            // 记录查询操作日志
            logOperation(cipherData, operateTable, operateField);

            return AjaxResult.success(dataMap);
        } catch (Exception e) {
            String cipherData = params != null && params.get("cipherText") != null ? params.get("cipherText").toString() : null;
            logOperation(cipherData, "t_user_sensitive_crypto", String.valueOf(params.get("field")));
            return AjaxResult.error("查询失败: " + e.getMessage());
        }
    }

    /**
     * 解密数据中的敏感字段
     * 支持多字段解密，使用逗号分隔字段名
     *
     * @param dataMap 数据Map
     */
    private void decryptSensitiveFields(Map<String, String> dataMap) {
        if (dataMap == null ) {
            return;
        }
//        循环dataMap

        for (Map.Entry<String, String> entry : dataMap.entrySet()) {
            try {
                // 对字段值进行解密
                String decryptedValue = SubstitutionCipherUtil.decrypt((String) entry.getValue());
                // 存储解密后的值到新字段，保留原始密文字段
                dataMap.put("decrypted", decryptedValue);
            } catch (Exception e) {
                // 解密失败时记录错误，但不影响整体处理
                System.err.println("解密字段" + entry.getKey() + "失败: " + e.getMessage());
            }
        }
    }

    /**
     * 调用远程服务进行敏感数据查询
     * 通过LogicService调用af-system服务查询加密的敏感数据
     *
     * @param params 查询参数
     * @return 查询结果（包含密文数据）
     */
    private Map<String, Object> callRemoteSensitiveDataQuery(Map<String, Object> params) {
        // 准备调用参数
        Map<String, Object> callParams = new HashMap<>(params);
        // 添加调用信息
        callParams.put("operatorId", String.valueOf(SecurityUtils.getUserId()));
        callParams.put("operatorName", SecurityUtils.getUsername());

        // 调用远程服务获取密文数据
        return (Map<String, Object>) logicService.remoteRun(ServiceNameConstants.SYSTEM_SERVICE, "sensitiveDataQuery", (JSONObject) callParams);
    }

    /**
     * 记录操作日志到远程服务
     * 根据新的审计日志表结构t_audit_log记录敏感数据操作
     *
     * @param cipherData 涉及的密文数据
     * @param operateTable 操作的表名
     * @param operateFields 操作的字段名
     */
    private void logOperation(String cipherData, String operateTable, String operateFields) {
        try {
            // 准备符合t_audit_log表结构的参数
            Map<String, Object> logParams = new HashMap<>();
            logParams.put("f_operator_id", String.valueOf(SecurityUtils.getUserId()));
            logParams.put("f_operator_name", SecurityUtils.getUsername());
            logParams.put("f_cipher_data", cipherData);
            logParams.put("f_operate_table", operateTable);
            logParams.put("f_operate_fields", operateFields);
            // 获取当前时间
            LocalDateTime now = LocalDateTime.now();
            logParams.put("f_operator_time", now.toString());
            logParams.put("f_module_name", "af-system");
            JSONObject logParamsJson = new JSONObject(logParams);
            // 异步调用日志记录服务，不影响主流程
            logicService.remoteRun(ServiceNameConstants.SYSTEM_SERVICE, "recordAuditLog", logParamsJson);
        } catch (Exception e) {
            // 记录日志失败不影响主流程，仅打印错误日志
            System.err.println("记录操作日志失败: " + e.getMessage());
        }
    }



//    /**
//     * 获取客户端IP地址
//     * @return IP地址字符串
//     */
//    private String getClientIp() {
//        return ServletUtils.getClientIP();
//    }
}
