package com.af.v4.system.runtime.dto;

import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.datasource.dialects.Dialect;
import com.af.v4.system.common.datasource.enums.DbType;
import com.af.v4.system.runtime.utils.DataTypeUtils;

/**
 * UpgradeField
 *
 * @author 张振宇
 */
public record UpgradeField(String tableName, String field, String type, String defaultValue) {

    public static final String DEFAULT_NOW_DATE = "now";

    /**
     * 构造函数
     */
    public UpgradeField {
        // 如果defaultValue为null，则保持为null
    }

    /**
     * 兼容旧版本的构造函数
     */
    public UpgradeField(String tableName, String field, String type) {
        this(tableName, field, type, null);
    }

    public String getSearchSql() {
        DbType dbType = DynamicDataSource.getDbType();
        Dialect dialect = dbType.dialect;

        String tableName = DataTypeUtils.getTableNameWithCase(tableName(), dialect);
        String fieldName = DataTypeUtils.getFieldNameWithCase(field(), dialect);
        if (dialect.isOracleFamily()) {
            return STR."select 1 from user_tab_columns where table_name = '\{tableName}' and column_name = '\{fieldName}'";
        } else if (dialect.isPostgresFamily()) {
            return STR."select 1 from information_schema.columns where table_name = '\{tableName}' and column_name = '\{fieldName}'";
        } else if (dialect.isMySqlFamily()) {
            return STR."select 1 from information_schema.columns where table_name = '\{tableName}' and column_name = '\{fieldName}'";
        } else if (dialect.isSQLServerFamily()) {
            return STR."select 1 from syscolumns where id = object_id('\{tableName}') and name = '\{fieldName}'";
        } else {
            throw new UnsupportedOperationException(STR."不支持的数据库类型: \{dbType}");
        }
    }

    public String getAddSql() {
        DbType dbType = DynamicDataSource.getDbType();
        Dialect dialect = dbType.dialect;
        String convertedType = DataTypeUtils.convertDataType(type, dbType);
        String tableName = DataTypeUtils.getTableNameWithCase(tableName(), dialect);
        String fieldName = DataTypeUtils.getFieldNameWithCase(field(), dialect);

        StringBuilder sql = new StringBuilder();

        if (dialect.isOracleFamily()) {
            sql.append("alter table ").append(tableName.toUpperCase())
                    .append(" add ").append(fieldName)
                    .append(" ").append(convertedType);
        } else if (dialect.isPostgresFamily()) {
            sql.append("alter table ").append(tableName.toLowerCase())
                    .append(" add column ").append(fieldName)
                    .append(" ").append(convertedType);
        } else if (dialect.isMySqlFamily()) {
            sql.append("alter table ").append(tableName)
                    .append(" add column ").append(fieldName)
                    .append(" ").append(convertedType);
        } else if (dialect.isSQLServerFamily()) {
            sql.append("alter table ").append(tableName)
                    .append(" add ").append(fieldName)
                    .append(" ").append(convertedType);
        } else {
            throw new UnsupportedOperationException(STR."不支持的数据库类型: \{dbType}");
        }

        // 如果指定了默认值，则添加默认值子句
        if (defaultValue != null && !defaultValue.isEmpty()) {
            sql.append(" default ");

            // 如果默认值为DEFAULT_NOW_DATE，则根据数据库类型获取当前时间函数
            if (DEFAULT_NOW_DATE.equalsIgnoreCase(defaultValue)) {
                if (dialect.isOracleFamily()) {
                    sql.append("SYSDATE");
                } else if (dialect.isPostgresFamily()) {
                    sql.append("CURRENT_TIMESTAMP");
                } else if (dialect.isMySqlFamily()) {
                    sql.append("NOW()");
                } else if (dialect.isSQLServerFamily()) {
                    sql.append("GETDATE()");
                } else {
                    sql.append("CURRENT_TIMESTAMP"); // 默认情况
                }
            } else {
                sql.append(defaultValue);
            }
        }

        return sql.toString();
    }
}
