package com.af.v4.system.common.workflow;

import org.json.JSONArray;
import org.json.JSONObject;
import java.util.*;

/**
 * FlowPathFinder
 * ----------------------------------------------------------
 * 从流程图（stepMap）和已执行节点（extraData）推导完整通路
 */
public class FlowPathFinder {

    /**
     * 获取完整路径
     *
     * @param stepMap   流程定义图，每个 key 为节点 id
     * @param extraData 已执行节点
     * @return JSONArray 完整可通行路径
     */
    public static JSONArray findPath(JSONObject stepMap, JSONArray extraData) {
        JSONArray path = new JSONArray();
        Set<String> visited = new HashSet<>();

        // 1. 清理 extraData 中的回退
        LinkedHashMap<String, JSONObject> filteredSteps = new LinkedHashMap<>();
        for (int i = 0; i < extraData.length(); i++) {
            JSONObject step = extraData.getJSONObject(i);
            String id = String.valueOf(step.optInt("id"));

            if (filteredSteps.containsKey(id)) {
                // 删除 id 之后的节点（回退）
                List<String> toRemove = new ArrayList<>();
                boolean startRemove = false;
                for (String key : filteredSteps.keySet()) {
                    if (startRemove) toRemove.add(key);
                    if (key.equals(id)) startRemove = true;
                }
                for (String r : toRemove) filteredSteps.remove(r);
            }
            filteredSteps.put(id, step);
        }

        // 2. 从第一个节点开始，沿流程顺序生成完整路径
        if (filteredSteps.isEmpty()) return path;

        // 获取第一个节点 id
        String curId = String.valueOf(filteredSteps.values().iterator().next().optInt("id"));

        int i = 1;
        while (curId != null && !visited.contains(curId)) {
            JSONObject node = stepMap.optJSONObject(curId);
            if (node == null) break;

            // 如果 extraData 中有该节点，带上 handler/date/note 等信息
            JSONObject stepInfo = filteredSteps.getOrDefault(curId, new JSONObject());
            node.put("handler", stepInfo.optString("name"));
            node.put("date", stepInfo.optString("date"));
            node.put("note", stepInfo.optString("note"));
            node.put("deadline", stepInfo.optString("deadline"));
            node.put("status", stepInfo.opt("status"));
            node.put("f_data", stepInfo.opt("f_data"));

            node.put("sequence", i ++);
            path.put(node);
            visited.add(curId);

            // 获取下一个节点
            curId = getNextStep(stepMap, curId, stepInfo);
        }

        return path;
    }

    /**
     * 获取当前节点的下一个节点
     * 优先选择 type=submit 的动作，否则选择第一个动作
     */
    private static String getNextStep(JSONObject stepMap, String curId, JSONObject stepInfo) {
        JSONObject curStep = stepMap.optJSONObject(curId);
        if (curStep == null) return null;

        JSONObject props = curStep.optJSONObject("properties");
        if (props == null) return null;

        JSONArray actions = props.optJSONArray("actions");
        if (actions == null || actions.isEmpty()) return null;

        String type = curStep.optString("type");

        // 条件分支节点，优先使用 extraData 中 __expressionRs
        if ("condition".equalsIgnoreCase(type) && stepInfo != null) {
            String fData = stepInfo.optString("f_data");
            if (fData != null && !fData.isEmpty()) {
                JSONObject fDataJson = new JSONObject(fData);
                if (fDataJson.has("__expressionRs")) {
                    return String.valueOf(fDataJson.optInt("__expressionRs"));
                }
            }
        }

        // 普通节点，优先 submit
        for (int i = 0; i < actions.length(); i++) {
            JSONObject action = actions.getJSONObject(i);
            if ("submit".equalsIgnoreCase(action.optString("type"))) {
                return String.valueOf(action.optInt("to"));
            }
        }

        // 如果没有 submit，选择第一个动作
        return String.valueOf(actions.getJSONObject(0).optInt("to"));
    }
}
