package com.af.v4.system.common.socket.core.client;

import com.af.v4.system.common.socket.config.SocketClientConfigItem;
import com.af.v4.system.common.socket.core.channel.ChannelData;
import io.netty.bootstrap.Bootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelOption;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetSocketAddress;

/**
 * SOCKET客户端
 */
public abstract class SocketClient extends Bootstrap {

    protected final Logger LOGGER = LoggerFactory.getLogger(getClass());

    /**
     * 发起连接
     *
     * @param socketConfigItem socket配置项
     */
    public void connect(SocketClientConfigItem socketConfigItem) {
        String socketType = socketConfigItem.getType().getValue();
        LOGGER.info("Netty: 连接{}服务端[{}], 地址{}", socketType, socketConfigItem.getName(), socketConfigItem.getAddress());
        Bootstrap bootstrap = initBootstrap(socketConfigItem);
        try {
            bootstrap.option(ChannelOption.CONNECT_TIMEOUT_MILLIS, socketConfigItem.getConnectionTimeout() * 1000); // 设置连接超时时间
            ChannelFuture channelFuture = bootstrap.connect(new InetSocketAddress(socketConfigItem.getHost(), socketConfigItem.getPort())).sync();
            if (channelFuture.isSuccess()) {
                Channel channel = channelFuture.channel();
                channel.closeFuture().addListener((ChannelFutureListener) future -> {
                    //通过回调只关闭自己监听的channel
                    ClientManager.remove(future.channel().id());
                    future.channel().close();
                });
                ClientManager.add(channel.id().asLongText(), new ChannelData(socketConfigItem, channel, new JSONObject()));
                senMes(socketConfigItem);
            }
        } catch (Exception e) {
            LOGGER.error("连接{}服务端[{}]时发生异常：{}", socketType, socketConfigItem.getName(), e.getMessage());
            throw new RuntimeException("服务端连接失败", e);
        }
    }

    /**
     * 初始化netty引导实例
     *
     * @return 引导实例
     */
    protected abstract Bootstrap initBootstrap(SocketClientConfigItem socketConfigItem);

    /**
     * 销毁实例
     */
    public abstract void destroy();

    /**
     * 连接成功之后发送消息
     */
    public abstract void senMes(SocketClientConfigItem socketClientConfigItem);
}
