package com.af.v4.system.common.socket.config;

import java.util.Objects;

/**
 * SSL 配置
 */
public class SocketSslConfig {

    /**
     * 是否启用 SSL
     */
    private boolean enabled;

    /**
     * 是否需要校验证书（true=需要加载可信列表，false=信任所有）
     */
    private boolean verifyPeerCertificate;

    /**
     * 服务端是否需要双向认证
     */
    private boolean requireClientAuth;

    /**
     * 私钥/证书所在的密钥库位置
     */
    private String keyStorePath;

    /**
     * 密钥库类型，默认 JKS
     */
    private String keyStoreType = "JKS";

    /**
     * 密钥库密码
     */
    private String keyStorePassword;

    /**
     * 私钥密码；未设置时默认和密钥库密码一致
     */
    private String keyPassword;

    /**
     * 信任库位置
     */
    private String trustStorePath;

    /**
     * 信任库类型，默认 JKS
     */
    private String trustStoreType = "JKS";

    /**
     * 信任库密码
     */
    private String trustStorePassword;

    public boolean isEnabled() {
        return enabled;
    }

    public SocketSslConfig setEnabled(boolean enabled) {
        this.enabled = enabled;
        return this;
    }

    public boolean isVerifyPeerCertificate() {
        return verifyPeerCertificate;
    }

    public SocketSslConfig setVerifyPeerCertificate(boolean verifyPeerCertificate) {
        this.verifyPeerCertificate = verifyPeerCertificate;
        return this;
    }

    public boolean isRequireClientAuth() {
        return requireClientAuth;
    }

    public SocketSslConfig setRequireClientAuth(boolean requireClientAuth) {
        this.requireClientAuth = requireClientAuth;
        return this;
    }

    public String getKeyStorePath() {
        return keyStorePath;
    }

    public SocketSslConfig setKeyStorePath(String keyStorePath) {
        this.keyStorePath = keyStorePath;
        return this;
    }

    public String getKeyStoreType() {
        return keyStoreType;
    }

    public SocketSslConfig setKeyStoreType(String keyStoreType) {
        if (keyStoreType != null && !keyStoreType.isBlank()) {
            this.keyStoreType = keyStoreType;
        }
        return this;
    }

    public String getKeyStorePassword() {
        return keyStorePassword;
    }

    public SocketSslConfig setKeyStorePassword(String keyStorePassword) {
        this.keyStorePassword = keyStorePassword;
        return this;
    }

    public String getKeyPassword() {
        return keyPassword;
    }

    public SocketSslConfig setKeyPassword(String keyPassword) {
        this.keyPassword = keyPassword;
        return this;
    }

    public String getTrustStorePath() {
        return trustStorePath;
    }

    public SocketSslConfig setTrustStorePath(String trustStorePath) {
        this.trustStorePath = trustStorePath;
        return this;
    }

    public String getTrustStoreType() {
        return trustStoreType;
    }

    public SocketSslConfig setTrustStoreType(String trustStoreType) {
        if (trustStoreType != null && !trustStoreType.isBlank()) {
            this.trustStoreType = trustStoreType;
        }
        return this;
    }

    public String getTrustStorePassword() {
        return trustStorePassword;
    }

    public SocketSslConfig setTrustStorePassword(String trustStorePassword) {
        this.trustStorePassword = trustStorePassword;
        return this;
    }

    public boolean hasKeyStore() {
        return keyStorePath != null && !keyStorePath.isBlank();
    }

    public boolean hasTrustStore() {
        return trustStorePath != null && !trustStorePath.isBlank();
    }

    public char[] keyStorePasswordChars() {
        if (keyStorePassword == null) {
            return new char[0];
        }
        return keyStorePassword.toCharArray();
    }

    public char[] keyPasswordChars() {
        if (keyPassword != null) {
            return keyPassword.toCharArray();
        }
        return keyStorePasswordChars();
    }

    public char[] trustStorePasswordChars() {
        if (trustStorePassword == null) {
            return new char[0];
        }
        return trustStorePassword.toCharArray();
    }

    /**
     * 用于缓存 SSL 上下文的唯一键
     */
    public String identityKey() {
        return enabled + "-" + verifyPeerCertificate + "-" + requireClientAuth + "-" + keyStorePath + "-" + keyStoreType + "-" + trustStorePath + "-" + trustStoreType;
    }

    @Override
    public String toString() {
        return "SocketSslConfig{enabled=" + enabled + ", verifyPeerCertificate=" + verifyPeerCertificate + ", requireClientAuth=" + requireClientAuth + ", keyStorePath='" + keyStorePath + "', trustStorePath='" + trustStorePath + "'}";
    }

    @Override
    public int hashCode() {
        return Objects.hash(enabled, verifyPeerCertificate, requireClientAuth, keyStorePath, keyStoreType, trustStorePath, trustStoreType);
    }
}

