package com.af.v4.system.common.socket;

import com.af.v4.system.common.socket.config.SocketConfig;
import com.af.v4.system.common.socket.config.SocketConfigItem;
import com.af.v4.system.common.socket.config.SocketSslConfig;
import com.af.v4.system.common.socket.core.server.SocketServer;
import com.af.v4.system.common.socket.core.server.modbus.ModBusServer;
import com.af.v4.system.common.socket.core.server.mqtt.MqttServer;
import com.af.v4.system.common.socket.core.server.tcp.TcpServer;
import com.af.v4.system.common.socket.core.server.udp.UdpServer;
import com.af.v4.system.common.socket.core.ssl.SocketSslContextFactory;
import com.af.v4.system.common.socket.enums.SocketTypeEnum;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.net.InetSocketAddress;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

@Component
public class SocketServerManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(SocketServerManager.class);

    private static final Map<String, SocketConfigItem> socketConfigMap = new HashMap<>();
    private final SocketConfig socketConfig;
    private final List<SocketServer<?, ?>> socketServerList = new ArrayList<>();

    public SocketServerManager(SocketConfig socketConfig) {
        this.socketConfig = socketConfig;
    }

    public static SocketConfigItem getSocketConfigItemByPort(String port) {
        return socketConfigMap.get(port);
    }

    public static Optional<SocketConfigItem> getConfigItemByChannel(io.netty.channel.Channel ch) {
        try {
            String port = String.valueOf(((InetSocketAddress) ch.localAddress()).getPort());
            return Optional.ofNullable(socketConfigMap.get(port));
        } catch (Exception e) {
            return Optional.empty();
        }
    }

    /**
     * 启动socket服务端
     */
    @PostConstruct
    public void startAll() {
        List<SocketConfigItem> socketConfigItemList = socketConfig.getSocketServerList();
        if (socketConfigItemList != null) {
            socketConfigItemList.forEach(item -> {
                // 打印SSL配置信息（仅TCP协议）
                if (item.getType() == SocketTypeEnum.TCP) {
                    logSslConfig(item);
                }
                
                SocketServer<?, ?> socketServer = createSocketServer(item);
                socketServerList.add(socketServer);
                item.getPorts().forEach(port -> socketConfigMap.put(port.toString(), item));
                if (socketServer != null) {
                    socketServer.run(item);
                }
            });
        }
    }

    /**
     * 打印SSL配置信息
     */
    private void logSslConfig(SocketConfigItem item) {
        SocketSslConfig sslConfig = item.getSslConfig();
        if (sslConfig != null && sslConfig.isEnabled()) {
            if (sslConfig.hasKeyStore()) {
                List<String> keyAliases = SocketSslContextFactory.getCertificateAliases(sslConfig, true);
                LOGGER.info("服务端[{}] SSL已启用 - 密钥库路径: {}, 证书别名: {}", 
                        item.getName(), sslConfig.getKeyStorePath(), 
                        keyAliases.isEmpty() ? "无" : String.join(", ", keyAliases));
            }
            if (sslConfig.hasTrustStore()) {
                List<String> trustAliases = SocketSslContextFactory.getCertificateAliases(sslConfig, false);
                LOGGER.info("服务端[{}] SSL已启用 - 信任库路径: {}, 证书别名: {}", 
                        item.getName(), sslConfig.getTrustStorePath(), 
                        trustAliases.isEmpty() ? "无" : String.join(", ", trustAliases));
            }
        } else {
            LOGGER.warn("服务端[{}] SSL未启用 - 连接将使用明文传输，存在安全风险", item.getName());
        }
    }

    /**
     * 创建SocketServer实例
     */
    private SocketServer<?, ?> createSocketServer(SocketConfigItem item) {
        SocketServer<?, ?> socketServer = null;
        if (item.getType() == SocketTypeEnum.TCP) {
            socketServer = new TcpServer();
        } else if (item.getType() == SocketTypeEnum.UDP) {
            socketServer = new UdpServer();
        } else if (item.getType() == SocketTypeEnum.MODBUS_TCP) {
            socketServer = new ModBusServer();
        } else if (item.getType() == SocketTypeEnum.MQTT) {
            socketServer = new MqttServer(item);
        }
        return socketServer;
    }

    @PreDestroy
    public void destroyAll() {
        for (SocketServer<?, ?> socketServer : socketServerList) {
            socketServer.destroy();
        }
    }
}
