package com.af.v4.system.common.socket.core.server.tcp;

import com.af.v4.system.common.core.enums.OSType;
import com.af.v4.system.common.core.service.ApplicationService;
import com.af.v4.system.common.socket.SocketServerManager;
import com.af.v4.system.common.socket.config.SocketConfigItem;
import com.af.v4.system.common.socket.core.server.SocketServer;
import io.netty.bootstrap.AbstractBootstrap;
import io.netty.bootstrap.ServerBootstrap;
import io.netty.channel.ChannelInitializer;
import io.netty.channel.ChannelOption;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.ServerChannel;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.epoll.EpollServerSocketChannel;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.ServerSocketChannel;
import io.netty.channel.socket.SocketChannel;
import io.netty.channel.socket.nio.NioServerSocketChannel;
import io.netty.handler.logging.LogLevel;
import io.netty.handler.logging.LoggingHandler;
import io.netty.handler.timeout.IdleStateHandler;
import io.netty.util.concurrent.DefaultEventExecutorGroup;
import io.netty.util.concurrent.DefaultThreadFactory;
import io.netty.util.concurrent.EventExecutorGroup;

public abstract class AbstractTcpServer extends SocketServer<ServerBootstrap, ServerChannel> {
    private EventLoopGroup bossGroup;
    private EventLoopGroup workerGroup;
    // 独立业务线程池
    protected EventExecutorGroup businessGroup;
    @Override
    protected AbstractBootstrap<ServerBootstrap, ServerChannel> initBootstrap() {
        Class<? extends ServerSocketChannel> channelClass;
        if (ApplicationService.getOSType() == OSType.LINUX) {
            bossGroup = new EpollEventLoopGroup();
            workerGroup = new EpollEventLoopGroup();
            channelClass = EpollServerSocketChannel.class;
        } else {
            bossGroup = new NioEventLoopGroup();
            workerGroup = new NioEventLoopGroup();
            channelClass = NioServerSocketChannel.class;
        }
        // 根据核心数动态设置线程池大小
        final int cores = Runtime.getRuntime().availableProcessors();
        businessGroup = new DefaultEventExecutorGroup(cores * 2 + 1, new DefaultThreadFactory("socket-business-thread", true));

        return new ServerBootstrap()
                .group(bossGroup, workerGroup)
                .channel(channelClass)
                .option(ChannelOption.SO_REUSEADDR, true)
                // 配置 编码器、解码器、业务处理
                .childHandler(new ChannelInitializer<SocketChannel>() {
                    @Override
                    protected void initChannel(SocketChannel ch) {
                        Integer idle = SocketServerManager.getConfigItemByChannel(ch).map(SocketConfigItem::getIdleSeconds).orElse(60);
                        Boolean debug = SocketServerManager.getConfigItemByChannel(ch).map(SocketConfigItem::getDebugLogging).orElse(false);
                        if (LOGGER.isDebugEnabled() || debug) {
                            ch.pipeline().addFirst(new LoggingHandler(LogLevel.DEBUG));
                        }
                        ch.pipeline().addLast(new IdleStateHandler(0, 0, idle <= 0 ? 60 : idle));
                        setChildHandler(ch);
                    }
                })
                // tcp缓冲区
                .option(ChannelOption.SO_BACKLOG, 128)
                // 将网络数据积累到一定的数量后,服务器端才发送出去,会造成一定的延迟。希望服务是低延迟的,建议将TCP_NODELAY设置为true
                .childOption(ChannelOption.TCP_NODELAY, true)
                // 保持长连接
                .childOption(ChannelOption.SO_KEEPALIVE, useKeepAlive())
                .childOption(ChannelOption.SO_REUSEADDR, true);
    }

    protected boolean useKeepAlive() {
        return true;
    }

    /**
     * 设置业务处理器
     * @param ch 通道
     */
    protected abstract void setChildHandler(SocketChannel ch);

    /**
     * 销毁其他业务组
     */
    protected abstract void destroyOtherGroup();

    @Override
    public void destroy() {
        closeAllChannels();
        if (businessGroup != null) {
            businessGroup.shutdownGracefully();
        }
        if (workerGroup != null) {
            workerGroup.shutdownGracefully();
        }
        if (bossGroup != null) {
            bossGroup.shutdownGracefully();
        }
        destroyOtherGroup();
    }
}
