package com.af.v4.system.common.socket.core.server.udp;

import com.af.v4.system.common.core.enums.OSType;
import com.af.v4.system.common.core.service.ApplicationService;
import com.af.v4.system.common.socket.SocketServerManager;
import com.af.v4.system.common.socket.config.SocketConfigItem;
import com.af.v4.system.common.socket.core.channel.impl.DatagramPacketChannelHandler;
import com.af.v4.system.common.socket.core.server.SocketServer;
import io.netty.bootstrap.AbstractBootstrap;
import io.netty.bootstrap.Bootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelInitializer;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.epoll.EpollDatagramChannel;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.DatagramChannel;
import io.netty.channel.socket.nio.NioDatagramChannel;
import io.netty.handler.logging.LogLevel;
import io.netty.handler.logging.LoggingHandler;
import io.netty.util.concurrent.DefaultEventExecutorGroup;
import io.netty.util.concurrent.DefaultThreadFactory;
import io.netty.util.concurrent.EventExecutorGroup;

/**
 * UDP服务端
 */
public class UdpServer extends SocketServer<Bootstrap, Channel> {
    private EventLoopGroup workerGroup;
    // 独立业务线程池
    protected EventExecutorGroup businessGroup;
    @Override
    protected AbstractBootstrap<Bootstrap, Channel> initBootstrap() {
        Class<? extends DatagramChannel> channelClass;
        if (ApplicationService.getOSType() == OSType.LINUX) {
            workerGroup = new EpollEventLoopGroup();
            channelClass = EpollDatagramChannel.class;
        } else {
            workerGroup = new NioEventLoopGroup();
            channelClass = NioDatagramChannel.class;
        }
        // 根据核心数动态设置线程池大小
        final int cores = Runtime.getRuntime().availableProcessors();
        businessGroup = new DefaultEventExecutorGroup(cores * 2, new DefaultThreadFactory("socket-business-thread", true));

        return new Bootstrap()
                .group(workerGroup)
                .channel(channelClass)
                // 配置 编码器、解码器、业务处理
                .handler(new ChannelInitializer<DatagramChannel>() {
                    @Override
                    protected void initChannel(DatagramChannel ch) {
                        Boolean debug = SocketServerManager.getConfigItemByChannel(ch).map(SocketConfigItem::getDebugLogging).orElse(false);
                        if (LOGGER.isDebugEnabled() || debug) {
                            ch.pipeline().addFirst(new LoggingHandler(LogLevel.DEBUG));
                        }
                        ch.pipeline().addLast(businessGroup, new DatagramPacketChannelHandler());
                    }
                });
    }

    @Override
    public void destroy() {
        closeAllChannels();
        if (businessGroup != null) {
            businessGroup.shutdownGracefully();
        }
        if (workerGroup != null) {
            workerGroup.shutdownGracefully();
        }
    }
}
