package com.af.v4.system.common.security.auth;

import com.af.v4.system.api.model.LoginUser;
import com.af.v4.system.common.security.annotation.RequiresPermissions;
import com.af.v4.system.common.security.annotation.RequiresRoles;

/**
 * Token 权限验证工具类
 */
public class AuthUtil {
    /**
     * 底层的 AuthLogic 对象
     */
    public static final AuthLogic authLogic = new AuthLogic();

    /**
     * 敏感数据权限前缀
     */
    public static final String PERMISSION_SENSITIVE_PREFIX = "sensitive:view";

    /**
     * 查看所有敏感字段明文的权限
     */
    public static final String PERMISSION_SENSITIVE_VIEW_ALL = PERMISSION_SENSITIVE_PREFIX + ":all";

    /**
     * 会话注销
     */
    public static void logout() {
        authLogic.logout();
    }

    /**
     * 会话注销，根据指定Token
     *
     * @param token 指定token
     */
    public static void logoutByToken(String token) {
        authLogic.logoutByToken(token);
    }

    /**
     * 检验当前会话是否已经登录，如未登录，则抛出异常
     */
    public static void checkLogin() {
        authLogic.checkLogin();
    }

    /**
     * 获取当前登录用户信息
     *
     * @param token 指定token
     * @return 用户信息
     */
    public static LoginUser getLoginUser(String token) {
        return authLogic.getLoginUser(token);
    }

    /**
     * 验证当前用户有效期
     *
     * @param loginUser 用户信息
     */
    public static void verifyLoginUserExpire(LoginUser loginUser) {
        authLogic.verifyLoginUserExpire(loginUser);
    }

    /**
     * 当前账号是否含有指定角色标识, 返回true或false
     * <p>注意：超级管理员自动拥有所有角色权限</p>
     *
     * @param role 角色标识
     * @return 是否含有指定角色标识
     */
    public static boolean hasRole(String role) {
        return authLogic.hasRole(role);
    }

    /**
     * 当前账号是否含有指定角色标识, 如果验证未通过，则抛出异常: NotRoleException
     *
     * @param role 角色标识
     */
    public static void checkRole(String role) {
        authLogic.checkRole(role);
    }

    /**
     * 根据注解传入参数鉴权, 如果验证未通过，则抛出异常: NotRoleException
     *
     * @param requiresRoles 角色权限注解
     */
    public static void checkRole(RequiresRoles requiresRoles) {
        authLogic.checkRole(requiresRoles);
    }

    /**
     * 当前账号是否含有指定角色标识 [指定多个，必须全部验证通过]
     *
     * @param roles 角色标识数组
     */
    public static void checkRoleAnd(String... roles) {
        authLogic.checkRoleAnd(roles);
    }

    /**
     * 当前账号是否含有指定角色标识 [指定多个，只要其一验证通过即可]
     *
     * @param roles 角色标识数组
     */
    public static void checkRoleOr(String... roles) {
        authLogic.checkRoleOr(roles);
    }

    /**
     * 当前账号是否含有指定权限, 返回true或false
     * <p>注意：拥有 *:*:* 权限的用户自动拥有所有权限</p>
     *
     * @param permission 权限码
     * @return 是否含有指定权限
     */
    public static boolean hasPermi(String permission) {
        return authLogic.hasPermi(permission);
    }

    /**
     * 当前账号是否含有指定权限, 如果验证未通过，则抛出异常: NotPermissionException
     *
     * @param permission 权限码
     */
    public static void checkPermi(String permission) {
        authLogic.checkPermi(permission);
    }

    /**
     * 根据注解传入参数鉴权, 如果验证未通过，则抛出异常: NotPermissionException
     *
     * @param requiresPermissions 权限注解
     */
    public static void checkPermi(RequiresPermissions requiresPermissions) {
        authLogic.checkPermi(requiresPermissions);
    }

    /**
     * 当前账号是否含有指定权限 [指定多个，必须全部验证通过]
     *
     * @param permissions 权限码数组
     */
    public static void checkPermiAnd(String... permissions) {
        authLogic.checkPermiAnd(permissions);
    }

    /**
     * 当前账号是否含有指定权限 [指定多个，只要其一验证通过即可]
     *
     * @param permissions 权限码数组
     */
    public static void checkPermiOr(String... permissions) {
        authLogic.checkPermiOr(permissions);
    }

    // ==================== 敏感数据权限检查 ====================

    /**
     * 检查敏感字段查看权限
     * <p>权限匹配规则（hasPermi 内部自动处理以下优先级）：</p>
     * <ol>
     *   <li>*:*:* - 全局所有权限（自动通过）</li>
     *   <li>sensitive:view:all - 查看所有敏感字段明文</li>
     *   <li>sensitive:view:* - 通配符匹配所有敏感字段</li>
     *   <li>sensitive:view:{fieldName} - 查看特定字段明文（如 sensitive:view:phone）</li>
     * </ol>
     *
     * @param fieldName 字段名（如：phone、idcard、email）
     * @return true=有权限查看明文，false=无权限（返回脱敏值）
     */
    public static boolean hasSensitiveViewPermission(String fieldName) {
        try {
            // 优先检查全局权限，再检查字段级权限
            return hasPermi(PERMISSION_SENSITIVE_VIEW_ALL) 
                || hasPermi(PERMISSION_SENSITIVE_PREFIX + ":" + fieldName);
        } catch (Exception e) {
            // 权限检查异常时，默认返回 false（安全优先）
            return false;
        }
    }
}
