package com.af.v4.system.common.resource.mapper;

import com.af.v4.system.common.core.proxy.liuli.ILiuLiConfigServiceProxy;
import com.af.v4.system.common.core.service.ApplicationService;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.datasource.enums.DbType;
import com.af.v4.system.common.resource.enums.ResourceType;
import org.json.JSONObject;
import org.springframework.stereotype.Component;

import java.util.Map;

/**
 * SQL映射器
 *
 * @author Mr.river
 */
@Component
public class SqlMapper extends AbstractResourceMapper<SqlMapper.SqlResource> {

    private static final Boolean DEFAULT_MOBILE_STATE = false;

    protected SqlMapper(ModuleMapper moduleMapper, ApplicationService applicationService, ILiuLiConfigServiceProxy liuLiConfigServiceProxy) {
        super(moduleMapper, applicationService, liuLiConfigServiceProxy);
    }

    @Override
    public ResourceType getResType() {
        return ResourceType.SQL;
    }

    @Override
    public String getFileName() {
        return "sql.xml";
    }

    @Override
    public String getFolderName() {
        return "sqls";
    }

    @Override
    protected SqlMapper.SqlResource buildResourceImpl(String moduleName, String key, String path, JSONObject liuliValue, Map<String, Object> attrMap) {
        // 所属数据源
        Object dataSource = attrMap.get("dataSource");
        // 是否支持手机端
        Object mobile = attrMap.get("mobile");
        return new SqlMapper.SqlResource(
                this,
                moduleName,
                key,
                path,
                liuliValue,
                dataSource == null ? null : dataSource.toString(),
                mobile == null ? DEFAULT_MOBILE_STATE : Boolean.parseBoolean(mobile.toString())
        );
    }

    @Override
    protected String rewriteSource(String source) {
        source = STR."$\{source}";
        //如果使用了新格式化参数写法，需要替换为解析语法树支持的写法
        if (source.contains("${")) {
            source = source.replace("${", "{");
        }
        return source;
    }

    @Override
    protected String[] getCandidateAliasList(String key) {
        DbType dbType = DynamicDataSource.getDbType();
        String alias = switch (dbType) {
            case oracle -> STR."\{key}_oracle";
            case mysql -> STR."\{key}_mysql";
            default -> null;
        };
        if (alias != null) {
            return new String[]{alias, key};
        } else {
            return new String[]{key};
        }
    }

    /**
     * Sql资源
     */
    public static class SqlResource extends AbstractResourceMapper.CommonResource implements AbstractResourceMapper.MobileResourceSupport, AbstractResourceMapper.DataSourceResourceSupport {
        /**
         * 所属数据源
         */
        private final String dataSource;
        /**
         * 是否支持手机端
         */
        private final boolean mobile;

        public SqlResource(AbstractResourceMapper<?> mapper, String moduleName, String key, String path, JSONObject liuliValue, String dataSource, Boolean mobile) {
            super(mapper, moduleName, key, path, liuliValue);
            this.dataSource = dataSource;
            this.mobile = mobile;
        }

        @Override
        public String getDataSource() {
            return dataSource;
        }

        @Override
        public boolean isMobile() {
            return mobile;
        }
    }
}
