package com.af.v4.system.common.redis.core;

import org.json.JSONObject;

import java.util.function.*;

/**
 * 链式注册器
 * <p>
 * 用于手动注册消费队列的回调逻辑（支持 onSuccess、onFailure、beforeRetry 等）
 * 支持链式调用设置处理逻辑及行为控制参数
 * <p>
 * 处理逻辑（如过滤器、重试条件、重试间隔）推荐由实现类中基于注解控制
 *
 */
public class ConsumerRegistration {

    /**
     * 队列名称
     */
    final String queueName;

    /**
     * 业务处理回调，返回业务结果
     */
    Function<JSONObject, Object> onProcess;

    /**
     * 成功消费回调
     */
    BiConsumer<JSONObject, Object> onSuccess;

    /**
     * 失败处理回调
     */
    BiConsumer<JSONObject, Exception> onFailure;

    /**
     * 重试前回调
     */
    BiConsumer<JSONObject, JSONObject> beforeRetry;

    /**
     * 当超过最大重试时是否重新入队
     */
    boolean requeueOnFailure = false;

    /**
     * 过滤器（通过配置注册的可在实现类中通过注解来控制）
     */
    Predicate<JSONObject> filter;

    /**
     * 重试条件
     */
    BiPredicate<JSONObject, Exception> condition;

    /**
     * 默认最大重试次数（若消息体未指定）
     */
    int defaultMaxRetries = 3;

    /**
     * 是否启用详细日志（默认启用）
     */
    boolean verbose = true;

    public ConsumerRegistration(String queueName) {
        this.queueName = queueName;
    }


    /**
     * 设置过滤器
     *
     * @param predicate 过滤条件
     * @return 当前注册器实例
     */
    public ConsumerRegistration filter(Predicate<JSONObject> predicate) {
        this.filter = predicate;
        return this;
    }

    /**
     * 设置重试条件
     *
     * @param predicate 条件函数（消息体 + 当前重试次数）
     * @return 当前注册器实例
     */
    public ConsumerRegistration condition(BiPredicate<JSONObject, Exception> predicate) {
        this.condition = predicate;
        return this;
    }

    /**
     * 设置成功消费回调函数（带业务返回值）
     * @param handler 成功回调处理器（参数：原始消息，业务返回值）
     * @return 当前注册器实例（链式调用）
     */
    public ConsumerRegistration onProcess(Function<JSONObject, Object> handler) {
        this.onProcess = handler;
        return this;
    }

    /**
     * 设置成功消费回调函数（带业务返回值）
     * @param handler 成功回调处理器（参数：原始消息，业务返回值）
     * @return 当前注册器实例（链式调用）
     */
    public ConsumerRegistration onSuccess(BiConsumer<JSONObject, Object> handler) {
        this.onSuccess = handler;
        return this;
    }

    /**
     * 设置失败处理回调函数
     *
     * @param handler 失败回调处理器（包含异常信息）
     * @return 当前注册器实例
     */
    public ConsumerRegistration onFailure(BiConsumer<JSONObject, Exception> handler) {
        this.onFailure = handler;
        return this;
    }

    /**
     * 设置重试前的处理函数
     *
     * @param handler 回调函数（含当前重试信息）
     * @return 当前注册器实例
     */
    public ConsumerRegistration beforeRetry(BiConsumer<JSONObject, JSONObject> handler) {
        this.beforeRetry = handler;
        return this;
    }

    /**
     * 设置失败后是否重新入队
     *
     * @param enable 是否启用
     * @return 当前注册器实例
     */
    public ConsumerRegistration requeueOnFailure(boolean enable) {
        this.requeueOnFailure = enable;
        return this;
    }

    /**
     * 设置默认最大重试次数（若队列实体中未设置）
     *
     * @param max 最大重试次数
     * @return 当前注册器实例
     */
    public ConsumerRegistration maxRetries(int max) {
        this.defaultMaxRetries = max;
        return this;
    }

    /**
     * 是否启用详细日志
     *
     * @param enable true 表示输出日志
     * @return 当前注册器实例
     */
    public ConsumerRegistration verbose(boolean enable) {
        this.verbose = enable;
        return this;
    }

    /**
     * 注册当前注册器至队列管理器中，启动多个消费线程
     *
     * @param manager 队列管理器
     * @param threads 启动线程数量
     */
    public void registerWith(FastCacheQueueManager manager, int threads) {
        for (int i = 1; i <= threads; i++) {
            manager.register(queueName, i, this);
        }
    }
}
